<?php

/**
 * 
 * @author Payment Plugins
 * @copyright 2016 Payment Plugins
 *
 */
class Braintree_Gateway_WC_Admin_Order_Actions
{

	public function __construct()
	{
		add_filter( 'woocommerce_order_actions', array (
				$this, 
				'add_order_actions' 
		) );
		add_action( 'woocommerce_order_action_braintree_void_transaction', array (
				$this, 
				'void_transaction' 
		) );
		add_action( 'woocommerce_order_action_braintree_submit_for_settlement', array (
				$this, 
				'submit_for_settlement' 
		) );
		add_filter( 'bulk_actions-edit-shop_order', array (
				$this, 
				'shop_order_bulk_actions' 
		), 50 );
		add_filter( 'handle_bulk_actions-edit-shop_order', array (
				$this, 
				'handle_shop_order_bulk_actions' 
		), 10, 3 );
	}

	public function add_order_actions( $actions )
	{
		global $theorder; // WC_Order object.
		global $braintree_transaction;
		
		if ( ! in_array( bwc_get_order_property( 'payment_method', $theorder ), bwc_get_payment_gateways() ) ) {
			return $actions; // This order is not a braintree order.
		}
		
		if ( get_class( $theorder ) !== 'WC_Order' ) {
			return $actions;
		}
		
		if ( $theorder->get_status() === 'failed' ) {
			return $actions; // payment failed so there is not a transaction id yet.
		}
		
		$id = $theorder->get_transaction_id();
		
		if ( empty( $id ) ) {
			return $actions;
		}
		
		try {
			if ( ! $braintree_transaction ) {
				$braintree_transaction = Braintree_Transaction::find( $id );
			}
			
			switch( $braintree_transaction->status ) {
				case Braintree_Transaction::AUTHORIZED :
					$actions [ 'braintree_submit_for_settlement' ] = __( 'Capture Authorized Transaction', 'braintree-payments' );
					$actions [ 'braintree_void_transaction' ] = __( 'Void Transaction', 'braintree-payments' );
					break;
				case Braintree_Transaction::SUBMITTED_FOR_SETTLEMENT :
					$actions [ 'braintree_void_transaction' ] = __( 'Void Transaction', 'braintree-payments' );
					break;
				default :
					break;
			}
		} catch( \Braintree\Exception $e ) {
			do_action( 'braintree_transaction_notfound_exception' );
			$message = sprintf( __( 'Transaction %s was not found in environment %s.', 'braintree-payments' ), $id, bt_manager()->get_environment() );
			bt_manager()->error( $message );
		}
		return $actions;
	}

	/**
	 * Void the transaction associated with the WC_Order.
	 *
	 * @param WC_Order $order        	
	 */
	public function void_transaction( $order )
	{
		$id = $order->get_transaction_id();
		try {
			$result = Braintree_Transaction::void( $id );
			if ( $result->success ) {
				$order->update_status( 'cancelled' );
				$order->add_order_note( sprintf( __( 'Transaction %s has been voided.', 'braintree-payments' ), $id ) );
			} else {
				$order->add_order_note( sprintf( __( 'There was an error voiding the transaction. Reason: %s', 'braintree-payments' ), $result->message ) );
			}
		} catch( \Braintree\Exception $e ) {
			$message = sprintf( __( 'Transaction %s could not be voided. Exception: %s', 'braintree-payments' ), $id, get_class( $e ) );
			$order->add_order_note( $message );
			bt_manager()->error( $message );
			do_action( 'braintree_transaction_void_exception', $e, $order );
		}
	}

	/**
	 * Submit the transaction associated with the WC_Order for settlement.
	 *
	 * @param WC_Order $order        	
	 */
	public function submit_for_settlement( $order )
	{
		$id = $order->get_transaction_id();
		$amount = $_POST [ 'braintree_settlement_amount' ];
		try {
			$result = Braintree_Transaction::submitForSettlement( $id, $amount );
			if ( $result->success ) {
				
				$order->set_total( $amount ); // update the order total with the
				                              // captured amount.
				$order->add_order_note( sprintf( __( 'Transaction %s was submitted for settlement in the amount of %s%s.', 'braintree-payments' ), $id, get_woocommerce_currency_symbol( bwc_get_order_property( 'order_currency', $order ) ), $amount ) );
			} else {
				$order->add_order_note( sprintf( __( 'There was an error submitting transaction %s for settlement. Reason: %s', 'braintree-payments' ), $id, $result->message ) );
			}
		} catch( \Braintree\Exception $e ) {
			$message = sprintf( __( 'There was an error submitting transaction %s for settlement. Exception: %s', 'braintree-payments' ), $id, get_class( $e ) );
			$order->add_order_note( $message );
			bt_manager()->error( $message );
			do_action( 'braintree_transaction_settle_exception', $e, $order );
		}
	}

	/**
	 *
	 * @since 2.6.28
	 * @param array $actions        	
	 */
	public function shop_order_bulk_actions( $actions )
	{
		global $wpdb;
		
		/*
		 * $query = $wpdb->prepare( "SELECT COUNT(*) AS count FROM $wpdb->posts AS posts INNER JOIN $wpdb->postmeta AS postmeta
		 * WHERE posts.post_type = 'shop_order' AND posts.ID = postmeta.post_id AND postmeta.meta_key = %s and meta_value = %s", '_transaction_status', Braintree_Transaction::AUTHORIZED );
		 * $results = $wpdb->get_var( $query );
		 *
		 * if ( $results > 0 ) {
		 * // only add the capture option if there are orders that have an authorized transaction.
		 * $actions [ 'bfwc_capture_charge' ] = __( 'Capture Charge', 'braintree-payments' );
		 * }
		 */
		
		// in future may add database query to check if there is an authorized status.
		$actions [ 'bfwc_capture_charge' ] = __( 'Capture Charge', 'braintree-payments' );
		$actions [ 'bfwc_void_transactions' ] = __( 'Void Transaction', 'braintree-payments' );
		
		return $actions;
	}

	/**
	 *
	 * @since 2.6.28
	 * @param string $redirect_to        	
	 * @param string $action        	
	 * @param array $ids        	
	 */
	public function handle_shop_order_bulk_actions( $redirect_to, $action, $ids )
	{
		if ( $action === 'bfwc_capture_charge' ) {
			foreach ( $ids as $id ) {
				$order = wc_get_order( $id );
				
				$transaction_id = $order->get_transaction_id();
				$status = get_post_meta( $id, '_transaction_status', true );
				
				// only attempt to capture if the order's transaction status is authorized.
				if ( $status === Braintree_Transaction::AUTHORIZED ) {
					try {
						$result = Braintree_Transaction::submitForSettlement( $transaction_id, $order->get_total() );
						
						if ( $result->success ) {
							update_post_meta( $id, '_transaction_status', $result->transaction->status );
							$order->add_order_note( __( 'Transaction successfully captured using bulk actions.', 'braintree-payments' ) );
							$order->payment_complete();
						} else {
							$order->add_order_note( sprintf( __( 'Transaction could not be captured. Reason: %s', 'braintree-payments' ), $result->message ) );
						}
					} catch( \Braintree\Exception $e ) {
						bt_manager()->error( sprintf( __( 'Error capturing order %s. Amount: %s. Reason: %s', 'braintree-payments' ), $id, $order->get_total(), $e->getMessage() ) );
						$order->add_order_note( sprintf( __( 'Transaction could not be captured. Reason: %s', 'braintree-payments' ), $e->getMessage() ) );
					}
				}
			}
		} elseif ( $action === 'bfwc_void_transactions' ) {
			foreach ( $ids as $id ) {
				$order = wc_get_order( $id );
				
				$transaction_id = $order->get_transaction_id();
				$status = get_post_meta( $id, '_transaction_status', true );
				
				switch( $status ) {
					case Braintree_Transaction::AUTHORIZED :
					case Braintree_Transaction::SUBMITTED_FOR_SETTLEMENT :
					case Braintree_Transaction::SETTLEMENT_PENDING :
						try {
							$result = Braintree_Transaction::void( $transaction_id );
							
							if ( $result->success ) {
								update_post_meta( $id, '_transaction_status', $result->transaction->status );
								$order->update_status( 'cancelled' );
							} else {
								$order->add_order_note( sprintf( __( 'Transaction %s could not be voided. Reason: %s', 'braintree-payments' ), $transaction_id, $result->message ) );
							}
						} catch( \Braintree\Exception $e ) {
						
						}
						break;
				}
			}
		}
		return $redirect_to;
	}
}
new Braintree_Gateway_WC_Admin_Order_Actions();