<?php
//==============================================================================
// Stripe Payment Gateway Pro v303.16
// 
// Author: Clear Thinking, LLC
// E-mail: johnathan@getclearthinking.com
// Website: http://www.getclearthinking.com
// 
// All code within this file is copyright Clear Thinking, LLC.
// You may not copy or reuse code within this file without written permission.
//==============================================================================

class ControllerExtensionPaymentStripe extends Controller {
	private $type = 'payment';
	private $name = 'stripe';
	
	public function logFatalErrors() {
		$error = error_get_last();
		if ($error && $error['type'] === E_ERROR) {
			$this->log->write('STRIPE PAYMENT GATEWAY: Order could not be completed due to the following fatal error:');
			$this->log->write('PHP Fatal Error:  ' . $error['message'] . ' in ' . $error['file'] . ' on line ' . $error['line']);
		}
	}
	
	//==============================================================================
	// index()
	//==============================================================================
	public function index() {
		register_shutdown_function(array($this, 'logFatalErrors'));
		
		$data['type'] = $this->type;
		$data['name'] = $this->name;
		$data['settings'] = $settings = $this->getSettings();
		
		// Check for currently uncaptured payments
		$today = date('Y-m-d');
		$last_check = (!empty($settings['uncaptured_check'])) ? $settings['uncaptured_check'] : 0;
		
		if (!empty($settings['uncaptured_emails']) && $today != $last_check) {
			$count = 0;
			$message = '<b>LIST OF CURRENTLY UNCAPTURED PAYMENTS</b><br><br>';
			
			$payment_intents_response = $this->curlRequest('GET', 'payment_intents', array('created' => array('lt' => time() - 3600), 'limit' => 100));
			
			foreach ($payment_intents_response['data'] as $payment_intent) {
				if ($payment_intent['status'] == 'requires_capture') {
					$count++;
					$message .= '<b>Payment ID:</b> <a target="_blank" href="https://dashboard.stripe.com/' . ($settings['transaction_mode'] == 'test' ? 'test/' : '') . 'payments/' . $payment_intent['id'] . '">' . $payment_intent['id'] . '</a><br>';
					$message .= '<b>Description:</b> ' . $payment_intent['description'] . '<br>';
					$message .= '<b>Expires:</b> ' . date('r', $payment_intent['created'] + 60*60*24*7) . '<br><br>';
				}
			}
			
			if ($count) {
				$admin_emails = explode(',', $settings['uncaptured_emails']);
				$subject = '[Stripe Payment Gateway] You have ' . $count . ' uncaptured payment(s) as of ' . $today;
				$this->sendEmail($admin_emails, $subject, $message);
			}
			
			$prefix = (version_compare(VERSION, '3.0', '<')) ? '' : $this->type . '_';
			$code = $prefix . $this->name;
			
			$this->db->query("DELETE FROM " . DB_PREFIX . "setting WHERE `code` = '" . $this->db->escape($code) . "' AND `key` = '" . $this->db->escape($code . '_uncaptured_check') . "'");
			$this->db->query("INSERT INTO " . DB_PREFIX . "setting SET `store_id` = 0, `code` = '" . $this->db->escape($code) . "', `key` = '" . $this->db->escape($code . '_uncaptured_check') . "', `value` = '" . $this->db->escape($today) . "', `serialized` = 0");
		}
		
		// Set up variables
		$data['language'] = $this->session->data['language'];
		$data['currency'] = $this->session->data['currency'];
		
		$main_currency = $this->db->query("SELECT * FROM " . DB_PREFIX . "setting WHERE `key` = 'config_currency' AND store_id = 0 ORDER BY setting_id DESC LIMIT 1")->row['value'];
		$decimal_factor = (in_array($data['currency'], array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF'))) ? 1 : 100;
		$data['decimal_factor'] = $decimal_factor;
		
		$data['error'] = '';
		$data['src_payment_intent'] = '';
		$data['checkout_success_url'] = $this->url->link('checkout/success', '', 'SSL');
		
		$data['stripe_errors'] = array(
			'card_declined',
			'expired_card',
			'incorrect_cvc',
			'incorrect_number',
			'incorrect_zip',
			'invalid_cvc',
			'invalid_expiry_month',
			'invalid_expiry_year',
			'invalid_number',
			'missing',
			'processing_error',
		);
		
		// Get order info
		$this->load->model('extension/' . $this->type . '/' . $this->name);
		$order_info = $this->{'model_extension_'.$this->type.'_'.$this->name}->getOrderInfo();
		
		// Sanitize order data
		$replace = array("'", "\n", "\r");
		
		$with = array("\'", ' ', ' ');
		
		foreach ($order_info as $key => &$value) {
			if (is_array($value)) {
				continue;
			}
			if ($key == 'email' || $key == 'telephone' || strpos($key, 'payment_') === 0 || strpos($key, 'shipping_') === 0) {
				$value = trim(str_replace($replace, $with, html_entity_decode($value, ENT_QUOTES, 'UTF-8')));
			}
			if ($key == 'telephone') {
				$value = substr($value, 0, 20);
			}
			if (empty($value)) {
				if ($key == 'payment_firstname') $value = 'none';
				if ($key == 'email') $value = 'no@email.com';
				if ($key == 'telephone') $value = 'none';
			}
		}
		
		$data['order_info'] = $order_info;
		
		// Set up other payment method data (Pro-specific)
		unset($this->session->data[$this->name . '_plans']);
		$plans = $this->getSubscriptionPlans($settings, $order_info);
		
		$data['country_code'] = $this->db->query("SELECT * FROM " . DB_PREFIX . "country WHERE country_id = " . (int)$this->config->get('config_country_id'))->row['iso_code_2'];
		$data['label'] = $this->request->server['HTTP_HOST'];
		$data['mapped_currency'] = $settings['currencies_' . $data['currency']];
		$data['mapped_amount'] = round($decimal_factor * $this->currency->convert($order_info['total'], $main_currency, $data['mapped_currency']));
		$data['order_amount'] = round($decimal_factor * $this->currency->convert($order_info['total'], $main_currency, $data['currency']));
		$data['store_url'] = $this->config->get('config_url');
		
		if (stripos($this->request->server['HTTP_USER_AGENT'], 'edg') !== false) {
			$user_agent = 'edge';
		} elseif (stripos($this->request->server['HTTP_USER_AGENT'], 'chrome') !== false || stripos($this->request->server['HTTP_USER_AGENT'], 'crios') !== false) {
			$user_agent = 'chrome';
		} elseif (stripos($this->request->server['HTTP_USER_AGENT'], 'safari') !== false) {
			$user_agent = 'safari';
		} else {
			$user_agent = 'other';
		}
		
		$data['other_payment_methods'] = $this->getOtherPaymentMethods($settings, $plans, $order_info['payment_iso_code_2'], $data['mapped_currency']);
		
		// Set payment request button line items
		$data['payment_request_line_items'] = array();
		
		foreach ($order_info['line_items'] as $line_item) {
			$data['payment_request_line_items'][] = array(
				'code'	=> $line_item['code'],
				'title'	=> str_replace("'", "\'", $line_item['title']),
				'value'	=> $this->currency->convert($line_item['value'], $main_currency, $data['currency']),
			);
		}
		
		// Set up Klarna items
		if (in_array('klarna', $data['other_payment_methods'])) {
			$data['klarna_items'] = array();
			
			// Get products
			if ($order_info['order_id']) {
				$order_products = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_product WHERE order_id = " . (int)$order_info['order_id'])->rows;
				foreach ($order_products as &$order_product) {
					$order_product['option'] = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_option WHERE order_product_id = " . (int)$order_product['order_product_id'])->rows;
				}
			} else {
				$order_products = $this->cart->getProducts();
			}
			
			// Add product options to name, and add to items array
			foreach ($order_products as $product) {
				$options = array();
				foreach ($product['option'] as $option) {
					$options[] = $option['name'] . ': ' . $option['value'];
				}
				if ($options) {
					$product['name'] .= ' (' . implode(', ', $options) . ')';
				}
				
				$data['klarna_items'][] = array(
					'type'			=> 'sku',
					'description'	=> str_replace("'", "\'", $product['name']),
					'quantity'		=> $product['quantity'],
					'amount'		=> round($decimal_factor * $product['total']),
				);
			}
			
			// Add Order Total lines to items array
			foreach ($order_info['line_items'] as $line_item) {
				if ($line_item['code'] == 'sub_total' || $line_item['code'] == 'total') continue;
				
				if ($line_item['code'] == 'shipping') {
					$item_type = 'shipping';
				} elseif (in_array($line_item['code'], array('tax', 'avalara_integration', 'taxcloud_integration', 'taxjar_integration'))) {
					$item_type = 'tax';
				} elseif ($line_item['value'] < 0) {
					$item_type = 'discount';
				} else {
					$item_type = 'sku';
				}
				
				$data['klarna_items'][] = array(
					'type'			=> $item_type,
					'description'	=> $line_item['title'],
					'quantity'		=> 1,
					'amount'		=> round($this->currency->convert($line_item['value'] * $decimal_factor, $main_currency, $data['mapped_currency'])),
				);
			}
		}
		
		// Find stripe_customer_id
		$data['customer'] = array();
		$data['logged_in'] = $this->customer->isLogged();
		$stripe_customer_id = '';
		
		if ($data['logged_in']) {
			$customer_id_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "stripe_customer WHERE customer_id = " . (int)$order_info['customer_id'] . " AND transaction_mode = '" . $this->db->escape($settings['transaction_mode']) . "'");
			
			if ($customer_id_query->num_rows) {
				$stripe_customer_id = $customer_id_query->row['stripe_customer_id'];
				
				if ($settings['allow_stored_cards']) {
					$payment_methods = $this->curlRequest('GET', 'payment_methods', array('customer' => $stripe_customer_id, 'type' => 'card'));
					
					if (!empty($payment_methods['error'])) {
						$this->log->write('STRIPE PAYMENT GATEWAY: ' . $payment_methods['error']['message']);
					} elseif ($data['settings']['allow_stored_cards']) {
						$data['customer_cards'] = $payment_methods['data'];
					}
					
					$customer_response = $this->curlRequest('GET', 'customers/' . $stripe_customer_id);
					
					$data['default_card'] = (!empty($customer_response['invoice_settings']['default_payment_method'])) ? $customer_response['invoice_settings']['default_payment_method'] : '';
				}
			}
		}
		
		// Stripe Checkout
		unset($this->session->data['stripe_checkout_session_id']);
		
		$negative_line_item = false;
		foreach ($order_info['line_items'] as $line_item) {
			if ($line_item['value'] < 0) {
				$negative_line_item = true;
			}
		}
		
		$data['use_stripe_checkout'] = $settings['checkout'];
		if (!empty($plans) && ($negative_line_item || $settings['prevent_guests'] && !$order_info['customer_id'])) {
			$data['use_stripe_checkout'] = false;
		}
		
		// Create Secure Remote Commerce PaymentIntent
		if (in_array('secure_remote_commerce', $data['other_payment_methods']) && !$data['use_stripe_checkout'] && $order_info['order_id']) {
			$payment_intent_data = array(
				'amount'				=> $data['mapped_amount'],
				'currency'				=> strtolower($data['mapped_currency']),
				'capture_method'		=> 'manual',
				'description'			=> $this->replaceShortcodes($settings['transaction_description'], $order_info),
				'metadata'				=> $this->metadata($order_info),
			);
			
			if ($stripe_customer_id) {
				$payment_intent_data['customer'] = $stripe_customer_id;
			}
			
			$data['src_payment_intent'] = $this->curlRequest('POST', 'payment_intents', $payment_intent_data);
			
			$this->session->data['src_payment_intent_id'] = $data['src_payment_intent']['id'];
		}
		
		// Render
		$theme = (version_compare(VERSION, '2.2', '<')) ? $this->config->get('config_template') : str_replace('theme_', '', $this->config->get('config_theme'));
		$template = (file_exists(DIR_TEMPLATE . $theme . '/template/extension/' . $this->type . '/' . $this->name . '.twig')) ? $theme : 'default';
		$template_file = DIR_TEMPLATE . $template . '/template/extension/' . $this->type . '/' . $this->name . '.twig';
		
		if (is_file($template_file)) {
			extract($data);
			
			ob_start();
			require(class_exists('VQMod') ? VQMod::modCheck(modification($template_file)) : modification($template_file));
			$output = ob_get_clean();
			
			return $output;
		} else {
			return 'Error loading template file';
		}
	}
	
	//==============================================================================
	// getSubscriptionPlans()
	//==============================================================================
	private function getSubscriptionPlans($settings, $order_info) {
		if (!empty($this->session->data[$this->name . '_plans'])) {
			return $this->session->data[$this->name . '_plans'];
		}
		
		$plans = array();
		
		if (empty($settings['subscriptions'])) {
			return $plans;
		}
		
		$cart_products = $this->cart->getProducts();
		$currency = $order_info['currency_code'];
		$decimal_factor = (in_array($settings['currencies_' . $currency], array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF'))) ? 1 : 100;
		
		foreach ($cart_products as $product) {
			$plan_id = '';
			$start_date = '';
			$cycles = 0;
			$product_name = $product['name'];
			
			foreach ($product['option'] as $option) {
				$product_name .= ' (' . $option['name'] . ': ' . $option['value'] . ')';
			}
			if (!empty($product['recurring']['name'])) {
				$product_name .= ' (' . $product['recurring']['name'] . ')';
			}
			
			if (!empty($settings['subscription_options'])) {
				foreach ($settings['subscription_options'] as $row) {
					foreach ($product['option'] as $option) {
						if (trim($option['name']) == trim($row['option_name']) && trim($option['value']) == trim($row['option_value']) && (empty($row['currency']) || $row['currency'] == $currency)) {
							$plan_id = trim($row['plan_id']);
							$start_date = $row['start_date'];
							$cycles = (int)$row['cycles'];
						}
					}
				}
			}
			
			if (!empty($product['recurring']) && !empty($settings['subscription_profiles'])) {
				foreach ($settings['subscription_profiles'] as $row) {
					if (trim($product['recurring']['name']) == trim($row['profile_name']) && (empty($row['currency']) || $row['currency'] == $currency)) {
						$plan_id = trim($row['plan_id']);
						$start_date = $row['start_date'];
						$cycles = (int)$row['cycles'];
					}
				}
			}
			
			if (empty($plan_id)) {
				$product_info = $this->db->query("SELECT * FROM " . DB_PREFIX . "product WHERE product_id = " . (int)$product['product_id'])->row;
				if (!empty($product_info['location'])) {
					$plan_id = trim($product_info['location']);
				}
			}
			
			if (empty($plan_id)) continue;
			
			// Get plan info
			$plan_response = $this->curlRequest('GET', 'plans/' . $plan_id);
			
			if (!empty($plan_response['error'])) continue;
			
			// Check coupons
			$coupon_code = '';
			$coupon_discount = 0;
			
			if (isset($this->session->data['coupon'])) {
				$coupon = (is_array($this->session->data['coupon'])) ? $this->session->data['coupon'][0] : $this->session->data['coupon'];
				
				$coupon_response = $this->curlRequest('GET', 'coupons/' . $coupon);
				
				if (empty($coupon_response['error'])) {
					$order_line_items = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_total WHERE order_id = " . (int)$order_info['order_id'] . " ORDER BY sort_order ASC")->rows;
					
					foreach ($order_line_items as $line_item) {
						if ($line_item['code'] == 'coupon' || $line_item['code'] == 'super_coupons' || $line_item['code'] == 'ultimate_coupons') {
							$coupon_code = $coupon;
							$coupon_discount = $line_item['value'];
						}
					}
				}
			}
			
			// Calculate tax rate
			$tax_rates = array();
			
			$tax_rates_response = $this->curlRequest('GET', 'tax_rates', array('limit' => 100));
			$opencart_tax_rates = $this->tax->getRates($product['total'], $product['tax_class_id']);
			
			foreach ($tax_rates_response['data'] as $stripe_tax_rate) {
				foreach ($opencart_tax_rates as $opencart_tax_rate) {
					if ($stripe_tax_rate['display_name'] == $opencart_tax_rate['name'] && (float)$stripe_tax_rate['percentage'] == (float)$opencart_tax_rate['rate']) {
						$tax_rates[] = $stripe_tax_rate;
					}
				}
			}
			
			if (empty($tax_rates)) {
				foreach ($opencart_tax_rates as $opencart_tax_rate) {
					$tax_rate_data = array(
						'display_name'	=> $opencart_tax_rate['name'],
						'inclusive'		=> 'false',
						'percentage'	=> $opencart_tax_rate['rate'],
					);
					
					$tax_rates_response = $this->curlRequest('POST', 'tax_rates', $tax_rate_data);
					
					if (!empty($tax_rates_response['error'])) {
						$this->log->write('STRIPE PAYMENT GATEWAY: Tax rate error: ' . $tax_rates_response['error']['message']);
					} else {
						$tax_rates[] = $tax_rates_response;
					}
				}
			}
			
			$overall_tax_rate = 0;
			$tax_rate_ids = array();
			
			foreach ($tax_rates as $tax_rate) {
				$overall_tax_rate += $tax_rate['percentage'];
				$tax_rate_ids[] = $tax_rate['id'];
			}
			
			// Add plan to array
			$plans[] = array(
				'cost'					=> $plan_response['amount'] / $decimal_factor,
				'coupon_code'			=> $coupon_code,
				'coupon_discount'		=> $coupon_discount,
				'currency'				=> $plan_response['currency'],
				'cycles'				=> $cycles,
				'id'					=> $plan_response['id'],
				'name'					=> (!empty($plan_response['nickname'])) ? $plan_response['nickname'] : 'Subscription',
				'product_id'			=> $product['product_id'],
				'product_key'			=> $product['product_id'] . json_encode($product['option']) . json_encode(!empty($product['recurring']) ? $product['recurring'] : array()),
				'product_name'			=> $product_name,
				'quantity'				=> $product['quantity'],
				'start_date'			=> $start_date,
				'taxed_cost'			=> $plan_response['amount'] / $decimal_factor * (1 + $overall_tax_rate / 100),
				'tax_rates'				=> $tax_rate_ids,
				'trial'					=> $plan_response['trial_period_days'],
				'shipping_cost'			=> 0,
				'taxed_shipping_cost'	=> 0,
				'total_plan_cost'		=> $plan_response['amount'] / $decimal_factor,
			);
		}
		
		// Check if shipping is required
		if (empty($settings['include_shipping']) || empty($order_info['shipping_code'])) {
			$this->session->data[$this->name . '_plans'] = $plans;
			return $plans;
		}
		
		// Get plan shipping costs (Pro-specific)
		foreach ($plans as &$plan) {
			$country_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "country WHERE country_id = " . (int)$order_info['shipping_country_id']);
			$shipping_address = array(
				'firstname'		=> $order_info['shipping_firstname'],
				'lastname'		=> $order_info['shipping_lastname'],
				'company'		=> $order_info['shipping_company'],
				'address_1'		=> $order_info['shipping_address_1'],
				'address_2'		=> $order_info['shipping_address_2'],
				'city'			=> $order_info['shipping_city'],
				'postcode'		=> $order_info['shipping_postcode'],
				'zone'			=> $order_info['shipping_zone'],
				'zone_id'		=> $order_info['shipping_zone_id'],
				'zone_code'		=> $order_info['shipping_zone_code'],
				'country'		=> $order_info['shipping_country'],
				'country_id'	=> $order_info['shipping_country_id'],
				'iso_code_2'	=> $order_info['shipping_iso_code_2'],
			);
			
			// Remove ineligible products
			foreach ($cart_products as $product) {
				$key = $product['product_id'] . json_encode($product['option']) . json_encode(!empty($product['recurring']) ? $product['recurring'] : array());
				if ($key != $plan['product_key']) {
					$this->cart->remove($key);
				}
			}
			
			// Get shipping rates
			$shipping_methods = $this->db->query("SELECT * FROM " . DB_PREFIX . "extension WHERE `type` = 'shipping' ORDER BY `code` ASC")->rows;
			$prefix = (version_compare(VERSION, '3.0', '<')) ? '' : 'shipping_';
			
			foreach ($shipping_methods as $shipping_method) {
				if (!$this->config->get($prefix . $shipping_method['code'] . '_status')) continue;
				
				if (version_compare(VERSION, '2.3', '<')) {
					$this->load->model('shipping/' . $shipping_method['code']);
					$quote = $this->{'model_shipping_' . $shipping_method['code']}->getQuote($shipping_address);
				} else {
					$this->load->model('extension/shipping/' . $shipping_method['code']);
					$quote = $this->{'model_extension_shipping_' . $shipping_method['code']}->getQuote($shipping_address);
				}
				
				if (empty($quote)) continue;
				
				foreach ($quote['quote'] as $q) {
					if ($q['code'] != $order_info['shipping_code'] || empty($q['cost'])) continue;
					
					$plan['shipping_cost'] = $q['cost'];
					$plan['taxed_shipping_cost'] = $this->tax->calculate($q['cost'], $q['tax_class_id']);
					
					break;
				}
			}
			
			// Restore cart
			$this->cart->clear();
			foreach ($cart_products as $product) {
				$options = array();
				foreach ($product['option'] as $option) {
					if (isset($options[$option['product_option_id']])) {
						if (!is_array($options[$option['product_option_id']])) $options[$option['product_option_id']] = array($options[$option['product_option_id']]);
						$options[$option['product_option_id']][] = $option['product_option_value_id'];
					} else {
						$options[$option['product_option_id']] = (!empty($option['product_option_value_id'])) ? $option['product_option_value_id'] : $option['value'];
					}
				}
				$recurring_profile = (!empty($product['recurring']['recurring_id'])) ? $product['recurring']['recurring_id'] : 0;
				$this->cart->add($product['product_id'], $product['quantity'], $options, $recurring_profile);
			}
		}
		
		$this->session->data[$this->name . '_plans'] = $plans;
		return $plans;
	}
	
	//==============================================================================
	// getOtherPaymentMethods()
	//==============================================================================
	private function getOtherPaymentMethods($settings, $plans, $country_code, $currency_code) {
		$other_payment_methods = array();
		
		if (stripos($this->request->server['HTTP_USER_AGENT'], 'edg') !== false) {
			$user_agent = 'edge';
		} elseif (stripos($this->request->server['HTTP_USER_AGENT'], 'chrome') !== false || stripos($this->request->server['HTTP_USER_AGENT'], 'crios') !== false) {
			$user_agent = 'chrome';
		} elseif (stripos($this->request->server['HTTP_USER_AGENT'], 'safari') !== false) {
			$user_agent = 'safari';
		} else {
			$user_agent = 'other';
		}
		
		$payment_request_countries = array('AE','AT','AU','BE','BG','BR','CA','CH','CI','CR','CY','CZ','DE','DK','DO','EE','ES','FI','FR','GB','GR','GT','HK','HU','ID','IE','IN','IT','JP','LT','LU','LV','MT','MX','MY','NL','NO','NZ','PE','PH','PL','PT','RO','SE','SG','SI','SK','SN','TH','TT','US','UY');
		
		foreach (array('applepay', 'googlepay', 'microsoftpay', 'afterpay', 'alipay', 'bancontact', 'eps', 'fpx', 'giropay', 'ideal', 'klarna', 'multibanco', 'p24', 'secure_remote_commerce', 'sepa', 'sofort', 'wechat') as $payment_type) {
			if (empty($settings[$payment_type])) continue;
			
			if ($payment_type == 'applepay' && ($user_agent != 'safari' || !in_array($country_code, $payment_request_countries))) continue;
			
			if ($payment_type == 'googlepay' && ($user_agent != 'chrome' || !in_array($country_code, $payment_request_countries))) continue;
			
			if ($payment_type == 'microsoftpay' && ($user_agent != 'edge' || !in_array($country_code, $payment_request_countries))) continue;
			
			if (!empty($plans)) continue;
			
			if ($payment_type == 'afterpay' && !in_array($country_code, array('AU', 'NZ', 'GB', 'US'))) continue;
			
			if ($payment_type == 'alipay' && !in_array($currency_code, array('CNY', 'AUD', 'CAD', 'EUR', 'GPB', 'HKD', 'JPY', 'MYR', 'NZD', 'SGD', 'USD'))) continue;
			
			if (in_array($payment_type, array('bancontact', 'eps', 'giropay', 'ideal', 'multibanco', 'sepa', 'sofort')) && $currency_code != 'EUR') continue;
			
			if ($payment_type == 'fpx' && $currency_code != 'MYR') continue;
			
			if ($payment_type == 'klarna') {
				if (!in_array($country_code, array('AT', 'BE', 'DK', 'FI', 'FR', 'DE', 'IE', 'IT', 'NL', 'NO', 'ES', 'SE', 'GB', 'US'))) continue;
				if ($country_code == 'AT' && $currency_code != 'EUR') continue;
				if ($country_code == 'BE' && $currency_code != 'EUR') continue;
				if ($country_code == 'DK' && $currency_code != 'DKK') continue;
				if ($country_code == 'FI' && $currency_code != 'EUR') continue;
				if ($country_code == 'FR' && $currency_code != 'EUR') continue;
				if ($country_code == 'DE' && $currency_code != 'EUR') continue;
				if ($country_code == 'IE' && $currency_code != 'EUR') continue;
				if ($country_code == 'NT' && $currency_code != 'EUR') continue;
				if ($country_code == 'NL' && $currency_code != 'EUR') continue;
				if ($country_code == 'NO' && $currency_code != 'NOK') continue;
				if ($country_code == 'ES' && $currency_code != 'EUR') continue;
				if ($country_code == 'SE' && $currency_code != 'SEK') continue;
				if ($country_code == 'GB' && $currency_code != 'GBP') continue;
				if ($country_code == 'US' && $currency_code != 'USD') continue;
			}
			
			if ($payment_type == 'p24' && !in_array($currency_code, array('EUR', 'PLN'))) continue;
			
			if ($payment_type == 'sepa' && !in_array($country_code, array('AT', 'BE', 'DE', 'ES', 'FR', 'IE', 'IT', 'LU', 'NL', 'PT'))) continue;
			
			if ($payment_type == 'sofort' && !in_array($country_code, array('AT', 'BE', 'DE', 'ES', 'IT', 'NL'))) continue;
			
			if ($payment_type == 'wechat' && !in_array($currency_code, array('AUD', 'CAD', 'CNY', 'EUR', 'GBP', 'HKD', 'JPY', 'SGD', 'USD'))) continue;
			
			$other_payment_methods[] = $payment_type;
		}
		
		return $other_payment_methods;
	}
	
	//==============================================================================
	// createCheckoutSession()
	//==============================================================================
	public function createCheckoutSession() {
		$quick_buy = !empty($this->request->post['quick_buy']);
		
		// Check for order_id
		$language = $this->session->data['language'];
		
		if (empty($this->session->data['order_id'])) {
			$json = array('error' => $settings['checkout_no_order_id_' . $language]);
			$this->response->addHeader('Content-Type: application/json');
			$this->response->setOutput(json_encode($json));
			return;
		}
		
		// Set up variables
		$settings = $this->getSettings();
		
		$this->load->model('extension/' . $this->type . '/' . $this->name);
		$order_info = $this->{'model_extension_'.$this->type.'_'.$this->name}->getOrderInfo();
		
		unset($this->session->data[$this->name . '_plans']);
		$plans = $this->getSubscriptionPlans($settings, $order_info);
		
		$currency = $this->session->data['currency'];
		$main_currency = $this->db->query("SELECT * FROM " . DB_PREFIX . "setting WHERE `key` = 'config_currency' AND store_id = 0 ORDER BY setting_id DESC LIMIT 1")->row['value'];
		$mapped_currency = $settings['currencies_' . $currency];
		$decimal_factor = (in_array($currency, array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF'))) ? 1 : 100;
		
		$stripe_customer_id = '';
		if ($this->customer->isLogged()) {
			$customer_id_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "stripe_customer WHERE customer_id = " . (int)$order_info['customer_id'] . " AND transaction_mode = '" . $this->db->escape($settings['transaction_mode']) . "'");
			if ($customer_id_query->num_rows) {
				$stripe_customer_id = $customer_id_query->row['stripe_customer_id'];
			}
		}
		
		// Get enabled payment methods
		$country_code = ($quick_buy) ? $order_info['shipping_iso_code_2'] : $order_info['payment_iso_code_2'];
		$other_payment_methods = $this->getOtherPaymentMethods($settings, $plans, $country_code, $mapped_currency);
		
		$payment_method_types = array('card');
		
		if (empty($plans)) {
			// Also currently supports: acss_debit, bacs_debit, boleto, grabpay, oxxo
			foreach (array('afterpay', 'alipay', 'bancontact', 'eps', 'fpx', 'giropay', 'ideal', 'klarna', 'p24', 'sepa', 'sofort', 'wechat') as $payment_type) {
				if (!in_array($payment_type, $other_payment_methods)) {
					continue;
				} elseif ($payment_type == 'afterpay') {
					if (!empty($order_info['shipping_firstname'])) {
						$payment_method_types[] = 'afterpay_clearpay';
					}
				} elseif ($payment_type == 'sepa') {
					$payment_method_types[] = 'sepa_debit';
				} elseif ($payment_type == 'wechat') {
					$payment_method_types[] = 'wechat_pay';
				} else {
					$payment_method_types[] = $payment_type;
				}
			}
		}
		
		// Set up checkout session data
		$checkout_data = array(
			'mode'					=> ($plans) ? 'subscription' : 'payment',
			'client_reference_id'	=> $order_info['order_id'],
			'line_items'			=> array(),
			'success_url'			=> $this->url->link('extension/' . $this->type . '/' . $this->name . '/checkoutComplete', '', 'SSL'),
			'cancel_url'			=> $this->url->link('checkout/cart', '', 'SSL'),
			'metadata'				=> array('order_id' => $order_info['order_id']),
			'payment_method_types'	=> $payment_method_types,
		);
		
		if ($quick_buy) {
			$checkout_data['metadata']['quick_buy'] = 'yes';
		}
		
		if (in_array('wechat_pay', $payment_method_types)) {
			$checkout_data['payment_method_options']['wechat_pay']['client'] = 'web';
		}
		
		if (!empty($settings['checkout_billing_address']) || empty($order_info['payment_firstname']) || $quick_buy) {
			$checkout_data['billing_address_collection'] = 'required';
		}
		
		if (!empty($settings['checkout_phone_number']) || $quick_buy) {
			$checkout_data['phone_number_collection'] = array('enabled' => 'true');
		}
		
		if (empty($plans)) {
			$checkout_data['payment_intent_data'] = array(
				'description'	=> $this->replaceShortcodes($settings['transaction_description'], $order_info),
				'metadata'		=> $this->metadata($order_info),
			);
			
			if (!empty($order_info['shipping_firstname'])) {
				$checkout_data['payment_intent_data']['shipping'] = array(
					'name'		=> $order_info['shipping_firstname'] . ' ' . $order_info['shipping_lastname'],
					'phone'		=> $order_info['telephone'],
					'address'	=> array(
						'line1'			=> $order_info['shipping_address_1'],
						'line2'			=> $order_info['shipping_address_2'],
						'city'			=> $order_info['shipping_city'],
						'state'			=> $order_info['shipping_zone'],
						'postal_code'	=> $order_info['shipping_postcode'],
						'country'		=> $order_info['shipping_iso_code_2'],
					),
				);
			}
			
			if ($settings['charge_mode'] == 'authorize') {
				$checkout_data['payment_intent_data']['capture_method'] = 'manual';
			}
		} elseif (empty($stripe_customer_id) && $this->customer->isLogged()) {
			// Set up billing address and shipping info
			if (empty($order_info['payment_firstname'])) {
				$billing_address = array();
			} else {
				$billing_address = array(
					'line1'			=> trim(html_entity_decode($order_info['payment_address_1'], ENT_QUOTES, 'UTF-8')),
					'line2'			=> trim(html_entity_decode($order_info['payment_address_2'], ENT_QUOTES, 'UTF-8')),
					'city'			=> trim(html_entity_decode($order_info['payment_city'], ENT_QUOTES, 'UTF-8')),
					'state'			=> trim(html_entity_decode($order_info['payment_zone'], ENT_QUOTES, 'UTF-8')),
					'postal_code'	=> trim(html_entity_decode($order_info['payment_postcode'], ENT_QUOTES, 'UTF-8')),
					'country'		=> trim(html_entity_decode($order_info['payment_iso_code_2'], ENT_QUOTES, 'UTF-8')),
				);
			}
			
			if (empty($order_info['shipping_firstname'])) {
				$shipping_info = array();
			} else {
				$shipping_info = array(
					'name'		=> trim(html_entity_decode($order_info['shipping_firstname'] . ' ' . $order_info['shipping_lastname'], ENT_QUOTES, 'UTF-8')),
					'phone'		=> trim(html_entity_decode($order_info['telephone'], ENT_QUOTES, 'UTF-8')),
					'address'	=> array(
						'line1'			=> trim(html_entity_decode($order_info['shipping_address_1'], ENT_QUOTES, 'UTF-8')),
						'line2'			=> trim(html_entity_decode($order_info['shipping_address_2'], ENT_QUOTES, 'UTF-8')),
						'city'			=> trim(html_entity_decode($order_info['shipping_city'], ENT_QUOTES, 'UTF-8')),
						'state'			=> trim(html_entity_decode($order_info['shipping_zone'], ENT_QUOTES, 'UTF-8')),
						'postal_code'	=> trim(html_entity_decode($order_info['shipping_postcode'], ENT_QUOTES, 'UTF-8')),
						'country'		=> trim(html_entity_decode($order_info['shipping_iso_code_2'], ENT_QUOTES, 'UTF-8')),
					),
				);
			}
			
			// Create customer mapping
			$customer_data = array(
				'address'		=> $billing_address,
				'description'	=> $order_info['firstname'] . ' ' . $order_info['lastname'] . ' (' . 'customer_id: ' . $order_info['customer_id'] . ')',
				'email'			=> $order_info['email'],
				'name'			=> $order_info['firstname'] . ' ' . $order_info['lastname'],
				'phone'			=> $order_info['telephone'],
				'shipping'		=> $shipping_info,
			);
			
			$customer_response = $this->curlRequest('POST', 'customers', $customer_data);
			
			if (!empty($customer_response['error'])) {
				$json = array('error' => $customer_response['error']['message']);
				$this->response->addHeader('Content-Type: application/json');
				$this->response->setOutput(json_encode($json));
				return;
			} else {
				$stripe_customer_id = $customer_response['id'];
				$this->db->query("INSERT INTO " . DB_PREFIX . "stripe_customer SET customer_id = " . (int)$order_info['customer_id'] . ", stripe_customer_id = '" . $this->db->escape($stripe_customer_id) . "', transaction_mode = '" . $this->db->escape($settings['transaction_mode']) . "'");
			}
		}
		
		if ($stripe_customer_id) {
			$checkout_data['customer'] = $stripe_customer_id;
		} elseif (!empty($order_info['email'])) {
			$checkout_data['customer_email'] = $order_info['email'];
		}
		
		// Check for negative line items
		$negative_line_item = false;
		foreach ($order_info['line_items'] as $line_item) {
			if ($line_item['value'] < 0) {
				$negative_line_item = true;
			}
		}
		
		// Set product line items
		if ($negative_line_item) {
			// Stripe does not support negative line items, so if a discount is present the extension can only send the order total
			$checkout_data['line_items'] = array(array(
				'price_data'	=> array(
					'currency'		=> strtolower($mapped_currency),
					'unit_amount'	=> round($decimal_factor * $this->currency->convert($order_info['total'], $main_currency, $mapped_currency)),
					'product_data'	=> array(
						'name'		=> html_entity_decode($settings['checkout_total_' . $language], ENT_QUOTES, 'UTF-8'),
						'images'	=> array(),
					),
				),
				'quantity'	=> 1,
			));
		} else {
			// No discounts are present, so line items can be shown
			if ($order_info['order_id'] && empty($plans)) {
				$order_products = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_product WHERE order_id = " . (int)$order_info['order_id'])->rows;
				foreach ($order_products as &$order_product) {
					$order_product['option'] = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_option WHERE order_product_id = " . (int)$order_product['order_product_id'])->rows;
				}
			} else {
				$order_products = $this->cart->getProducts();
			}
			
			foreach ($order_products as $product) {
				$product['name'] = str_replace(array('[', ']'), '', $product['name']);
				
				// Add product options to name
				$options = array();
				foreach ($product['option'] as $opt) {
					$options[] = $opt['name'] . ': ' . $opt['value'];
				}
				if ($options) {
					$product['name'] .= ' (' . implode(', ', $options) . ')';
				}
				
				// Check whether product is mapped to a subscription
				$product_plan = false;
				
				foreach ($plans as $plan) {
					$product_key = $product['product_id'] . json_encode($product['option']) . json_encode(!empty($product['recurring']) ? $product['recurring'] : array());
					if ($product_key == $plan['product_key']) {
						$product_plan = $plan;
					}
				}
				
				// Skip products that have 0.00 prices because Stripe rejects them
				if ($product['price'] < 0.005) continue;
				
				// Add product to line items array
				if ($product_plan) {
					$checkout_data['line_items'][] = array(
						'price'		=> $plan['id'],
						'quantity'	=> $product['quantity'],
						'tax_rates'	=> $plan['tax_rates'],
					);
					
					foreach ($order_info['line_items'] as &$reference_line_item) {
						if ($reference_line_item['code'] == 'tax') {
							$reference_line_item['value'] -= $plan['taxed_cost'] - $plan['cost'];
						}
					}
				} else {
					$product_info = $this->db->query("SELECT * FROM " . DB_PREFIX . "product WHERE product_id = " . (int)$product['product_id'])->row;
					if (!empty($product_info['image'])) {
						$image = HTTPS_SERVER . 'image/' . str_replace(' ', '%20', $product_info['image']);
						$image = preg_replace_callback('/[^\x20-\x5a]/', function($match) { return urlencode($match[0]); }, $image);
						$images = array($image);
					} else {
						$images = array();
					}
					
					$checkout_data['line_items'][] = array(
						'price_data'	=> array(
							'currency'		=> strtolower($mapped_currency),
							'unit_amount'	=> round($this->currency->convert($product['price'] * $decimal_factor, $main_currency, $mapped_currency)),
							'product_data'	=> array(
								'name'		=> html_entity_decode($product['name'], ENT_QUOTES, 'UTF-8'),
								'images'	=> $images,
							),
						),
						'quantity'	=> $product['quantity'],
					);
				}
			}
			
			// Set gift voucher line items
			if (!empty($this->session->data['vouchers'])) {
				foreach ($this->session->data['vouchers'] as $voucher) {
					$checkout_data['line_items'][] = array(
						'price_data'	=> array(
							'currency'		=> strtolower($mapped_currency),
							'unit_amount'	=> round($voucher['amount'] * $decimal_factor),
							'product_data'	=> array(
								'name'		=> html_entity_decode($voucher['description'], ENT_QUOTES, 'UTF-8'),
								'images'	=> array(),
							),
						),
						'quantity'	=> 1,
					);
				}
			}
			
			// Set Order Total line items
			foreach ($order_info['line_items'] as $line_item) {
				if ($line_item['code'] == 'sub_total' || $line_item['code'] == 'total' || empty($line_item['value'])) {
					continue;
				}
				
				$checkout_data['line_items'][] = array(
					'price_data'	=> array(
						'currency'		=> strtolower($mapped_currency),
						'unit_amount'	=> round($this->currency->convert($line_item['value'] * $decimal_factor, $main_currency, $mapped_currency)),
						'product_data'	=> array(
							'name'		=> html_entity_decode($line_item['title'], ENT_QUOTES, 'UTF-8'),
							'images'	=> array(),
						),
					),
					'quantity'	=> 1,
				);
			}
		}
		
		// Set checkout session
		$checkout_session = $this->curlRequest('POST', 'checkout/sessions', $checkout_data);
		
		if (!empty($checkout_session['error'])) {
			$json = array(
				'error'			=> $checkout_session['error']['message'],
			);
		} else {
			$this->session->data['stripe_checkout_session_id'] = $checkout_session['id'];
			
			$json = array(
				'key'			=> $settings[$settings['transaction_mode'] . '_publishable_key'],
				'account_id'	=> $settings['account_id'],
				'session_id'	=> $checkout_session['id'],
			);
		}
		
		$this->response->addHeader('Content-Type: application/json');
		$this->response->setOutput(json_encode($json));
	}
	
	//==============================================================================
	// deleteCard()
	//==============================================================================
	public function deleteCard() {
		if (empty($this->request->post['card_id'])) return;
		
		$detach_response = $this->curlRequest('POST', 'payment_methods/' . $this->request->post['card_id'] . '/detach');
		
		if (!empty($delete_response['error'])) {
			echo $delete_response['error']['message'];
		}
	}
	
	//==============================================================================
	// triggerError()
	//==============================================================================
	private function triggerError($datatype, $message, $order_id = 0) {
		if ($order_id) {
			$settings = $this->getSettings();
			
			$this->db->query("UPDATE `" . DB_PREFIX . "order` SET order_status_id = " . (int)$settings['error_status_id'] . " WHERE order_id = " . (int)$order_id);
			$this->db->query("INSERT INTO " . DB_PREFIX . "order_history SET order_id = " . (int)$order_id . ", order_status_id = " . (int)$settings['error_status_id'] . ", notify = 0, comment = '" . $this->db->escape($message) . "', date_added = NOW()");
			
			$subject = 'Payment Error for Order #' . $order_id;
			$message = 'When attempting to finalize payment for order #' . $order_id . ', Stripe reported the following error: "' . $message . '"';
			$this->sendEmail($this->config->get('config_email'), $subject, $message);
		} else {
			if ($datatype == 'json') {
				echo json_encode(array('errorMessage' => $message));
			} else {
				echo $message;
			}
		}
	}
	
	//==============================================================================
	// errorPage()
	//==============================================================================
	private function errorPage($message) {
		$settings = $this->getSettings();
		$language = (isset($this->session->data['language'])) ? $this->session->data['language'] : $this->config->get('config_language');
		
		$header = $this->load->controller('common/header');
		$footer = $this->load->controller('common/footer');
		
		$error_page = html_entity_decode($settings['error_page_' . $language], ENT_QUOTES, 'UTF-8');
		$error_page = str_replace(array('[header]', '[error]', '[footer]'), array($header, $message, $footer), $error_page);
		
		echo $error_page;
	}
	
	//==============================================================================
	// src()
	//==============================================================================
	public function src() {
		if ($this->request->get['mpstatus'] == 'cancel') {
			$this->response->redirect($this->url->link('checkout/checkout', '', 'SSL'));
			return;
		}
		
		$payment_intent_id = $this->session->data['src_payment_intent_id'];
		
		$confirmation_data = array(
			'payment_method_data' => array(
				'type'	=> 'card',
				'card'	=> array(
					'masterpass' => array(
						'cart_id'			=> str_replace('pi_', '', $payment_intent_id),
						'transaction_id'	=> $this->request->get['oauth_verifier'],
					),
				),
			),
		);
		
		$confirm_response = $this->curlRequest('POST', 'payment_intents/' . $payment_intent_id . '/confirm', $confirmation_data);
		
		if (!empty($confirm_response['error'])) {
			$this->errorPage($confirm_response['error']['message']);
		} else {
			$this->finalizePayment($payment_intent_id);
		}
	}
	
	//==============================================================================
	// createPaymentIntent()
	//==============================================================================
	public function createPaymentIntent($source = array()) {
		$settings = $this->getSettings();
		$language = (isset($this->session->data['language'])) ? $this->session->data['language'] : $this->config->get('config_language');
		
		// Check if customer has already exceeded the allowed number of payment attempts
		if (empty($source)) {
			if (empty($this->session->data[$this->name . '_payment_attempts'])) {
				$this->session->data[$this->name . '_payment_attempts'] = 1;
			} else {
				$this->session->data[$this->name . '_payment_attempts']++;
			}
			
			if (!empty($settings['attempts']) && $this->session->data[$this->name . '_payment_attempts'] > (int)$settings['attempts']) {
				$this->triggerError('json', $settings['attempts_exceeded_' . $language]);
				return;
			}
		}
		
		// Get order data
		if (empty($this->session->data['order_id'])) {
			$this->triggerError('json', 'Missing order_id');
			return;
		}
		
		$this->load->model('checkout/order');
		$order_id = $this->session->data['order_id'];
		$order_info = $this->model_checkout_order->getOrder($order_id);
		
		if (empty($order_info['email'])) {
			$this->triggerError('json', 'Please fill in your order information before attempting payment.');
			return;
		}
		
		$order_info['telephone'] = substr($order_info['telephone'], 0, 20);
		$source_order_id = (!empty($source)) ? $order_id : 0;
		
		// Check for second payment attempt on the same order
		$order_history_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_history WHERE order_id = " . (int)$order_id . " AND `comment` LIKE '%Stripe Payment ID%'");
		
		if ($order_history_query->num_rows) {
			$json = array(
				'payment_intent_id'	=> 'order_status_id_' . $order_history_query->row['order_status_id'],
				'status'			=> 'double_payment',
			);
				
			echo json_encode($json);
			return;
		}
		
		// Get subscription plan data
		$customer_id = $order_info['customer_id'];
		$plans = $this->getSubscriptionPlans($settings, $order_info);
		
		if (!empty($plans) && $settings['prevent_guests'] && !$customer_id) {
			$this->triggerError('json', $settings['text_customer_required_' . $language], $source_order_id);
			return;
		}
		
		// Check if payment is a non-card method
		$store_card = false;
		
		if (!empty($source)) {
			$payment_type = $source['type'];
		} elseif ($this->request->post['payment_method'] == 'afterpay') {
			$payment_type = 'afterpay_clearpay';
		} elseif ($this->request->post['payment_method'] == 'fpx') {
			$payment_type = 'fpx';
		} elseif ($this->request->post['payment_method'] == 'sepa') {
			$payment_type = 'sepa_debit';
		} else {
			$payment_type = 'card';
			$store_card = (!empty($plans) || (isset($this->request->post['store_card']) && $this->request->post['store_card'] == 'true') || $settings['send_customer_data'] == 'always');
		}
		
		// Set up billing address and shipping info
		$billing_address = array(
			'line1'			=> trim(html_entity_decode($order_info['payment_address_1'], ENT_QUOTES, 'UTF-8')),
			'line2'			=> trim(html_entity_decode($order_info['payment_address_2'], ENT_QUOTES, 'UTF-8')),
			'city'			=> trim(html_entity_decode($order_info['payment_city'], ENT_QUOTES, 'UTF-8')),
			'state'			=> trim(html_entity_decode($order_info['payment_zone'], ENT_QUOTES, 'UTF-8')),
			'postal_code'	=> trim(html_entity_decode($order_info['payment_postcode'], ENT_QUOTES, 'UTF-8')),
			'country'		=> trim(html_entity_decode($order_info['payment_iso_code_2'], ENT_QUOTES, 'UTF-8')),
		);
		
		if (empty($order_info['shipping_firstname'])) {
			$shipping_info = array();
		} else {
			$shipping_info = array(
				'name'		=> trim(html_entity_decode($order_info['shipping_firstname'] . ' ' . $order_info['shipping_lastname'], ENT_QUOTES, 'UTF-8')),
				'phone'		=> trim(html_entity_decode($order_info['telephone'], ENT_QUOTES, 'UTF-8')),
				'address'	=> array(
					'line1'			=> trim(html_entity_decode($order_info['shipping_address_1'], ENT_QUOTES, 'UTF-8')),
					'line2'			=> trim(html_entity_decode($order_info['shipping_address_2'], ENT_QUOTES, 'UTF-8')),
					'city'			=> trim(html_entity_decode($order_info['shipping_city'], ENT_QUOTES, 'UTF-8')),
					'state'			=> trim(html_entity_decode($order_info['shipping_zone'], ENT_QUOTES, 'UTF-8')),
					'postal_code'	=> trim(html_entity_decode($order_info['shipping_postcode'], ENT_QUOTES, 'UTF-8')),
					'country'		=> trim(html_entity_decode($order_info['shipping_iso_code_2'], ENT_QUOTES, 'UTF-8')),
				),
			);
		}
		
		// Create or update customer
		$customer_id_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "stripe_customer WHERE customer_id = " . (int)$customer_id . " AND transaction_mode = '" . $this->db->escape($settings['transaction_mode']) . "'");
		$stripe_customer_id = (!empty($customer_id_query->row['stripe_customer_id'])) ? $customer_id_query->row['stripe_customer_id'] : '';
		
		if ($store_card || $plans) {
			$customer_data = array(
				'address'		=> $billing_address,
				'description'	=> $order_info['firstname'] . ' ' . $order_info['lastname'] . ' (' . 'customer_id: ' . $order_info['customer_id'] . ')',
				'email'			=> $order_info['email'],
				'name'			=> $order_info['firstname'] . ' ' . $order_info['lastname'],
				'phone'			=> $order_info['telephone'],
				'shipping'		=> $shipping_info,
			);
			
			$customer_response = $this->curlRequest('POST', 'customers' . ($stripe_customer_id ? '/' . $stripe_customer_id : ''), $customer_data);
			
			if (!empty($customer_response['error'])) {
				$this->triggerError('json', $customer_response['error']['message'], $source_order_id);
				return;
			}
			
			if ($customer_id && !$stripe_customer_id) {
				$this->db->query("INSERT INTO " . DB_PREFIX . "stripe_customer SET customer_id = " . (int)$customer_id . ", stripe_customer_id = '" . $this->db->escape($customer_response['id']) . "', transaction_mode = '" . $this->db->escape($settings['transaction_mode']) . "'");
			}
			
			$stripe_customer_id = $customer_response['id'];
		}
		
		$this->session->data['stripe_customer_id'] = $stripe_customer_id;
		
		// Calculate amount
		$currency = $settings['currencies_' . $order_info['currency_code']];
		$main_currency = $this->db->query("SELECT * FROM " . DB_PREFIX . "setting WHERE `key` = 'config_currency' AND store_id = 0 ORDER BY setting_id DESC LIMIT 1")->row['value'];
		$decimal_factor = (in_array($currency, array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF'))) ? 1 : 100;
		
		$amount = $order_info['total'];
		
		if ($plans) {
			$amount -= $plans[0]['coupon_discount'];
			
			if (!empty($settings['merge_subscriptions'])) {
				foreach ($plans as $plan) {
					$amount -= $plan['taxed_cost'] * $plan['quantity'];
				}
				$amount -= $plans[0]['taxed_shipping_cost'];
			} else {
				foreach ($plans as $plan) {
					$amount -= $plan['taxed_cost'] * $plan['quantity'];
					$amount -= $plan['taxed_shipping_cost'];
				}
			}
		}
		
		// Set up payment intent data
		$json = array(
			'status'			=> '',
			'payment_intent_id'	=> '',
		);
		
		if ($amount >= 0.5) {
			$capture_method = ($payment_type == 'card') ? 'manual' : 'automatic';
			
			$curl_data = array(
				'amount'				=> round($decimal_factor * $this->currency->convert($amount, $main_currency, $currency)),
				'currency'				=> strtolower($currency),
				'capture_method'		=> $capture_method,
				'confirm'				=> 'true',
				'confirmation_method'	=> 'manual',
				'description'			=> $this->replaceShortcodes($settings['transaction_description'], $order_info),
				'metadata'				=> $this->metadata($order_info),
				'payment_method_types'	=> array($payment_type),
				'save_payment_method'	=> ($store_card) ? 'true' : 'false',
				'shipping'				=> $shipping_info,
			);
			
			if ($stripe_customer_id) {
				$curl_data['customer'] = $stripe_customer_id;
			}
			
			if ($settings['always_send_receipts']) {
				$curl_data['receipt_email'] = $order_info['email'];
			}
			
			if ($payment_type == 'card') {
				$curl_data['payment_method'] = $this->request->post['payment_method'];
				//$curl_data['payment_method_options']['card']['request_three_d_secure'] = 'any';
			} elseif (in_array($payment_type, array('afterpay_clearpay', 'fpx', 'sepa_debit'))) {
				unset($curl_data['capture_method']);
				unset($curl_data['confirm']);
				unset($curl_data['confirmation_method']);
				unset($curl_data['save_payment_method']);
			} else {
				$curl_data['source'] = $source['id'];
			}
			
			// Create payment intent
			$payment_intent = $this->curlRequest('POST', 'payment_intents', $curl_data);
			
			if (!empty($payment_intent['error'])) {
				// Add error info to order history
				$strong = '<strong style="display: inline-block; width: 155px; padding: 2px 5px">';
				$hr = '<hr style="margin: 5px">';
				$error = (!empty($payment_intent['error']['code'])) ? $payment_intent['error']['code'] : $payment_intent['error']['message'];
				
				$comment = $strong . 'Stripe Payment Error:</strong>' . $error . '<br>';
				
				if (!empty($payment_intent['error']['decline_code'])) {
					$comment .= $strong . 'Decline Code:</strong>' . $payment_intent['error']['decline_code'] . '<br>';
					
					if ($payment_intent['error']['decline_code'] == 'fraudulent' && !empty($settings['decline_code_emails'])) {
						$admin_emails = explode(',', $settings['decline_code_emails']);
						$subject = '[Stripe Payment Gateway] Fraudulent payment attempt by ' .  $order_info['email'] . ' for order ' . $order_info['order_id'];
						$message = 'Customer ' . $order_info['email'] . ' had a declined payment with the code "fraudulent" for order ' . $order_info['order_id'] . '. Check the order history in OpenCart to see the Stripe transaction data.';
						$this->sendEmail($admin_emails, $subject, $message);
					}
				}
				
				if (!empty($payment_intent['error']['payment_intent'])) {
					$pm = $payment_intent['error']['payment_intent']['last_payment_error']['payment_method'];
				} else {
					$pm = array('type' => $payment_type);
				}
				
				if (!empty($pm['billing_details'])) {
					$comment .= $hr . $strong . 'Billing Details:</strong>' . $pm['billing_details']['name'] . '<br>';
					if (!empty($pm['billing_details']['address'])) {
						$comment .= $strong . '&nbsp;</strong>' . $pm['billing_details']['address']['line1'] . '<br>';
						if (!empty($card_address['line2'])) $comment .= $strong . '&nbsp;</strong>' . $pm['billing_details']['address']['line2'] . '<br>';
						$comment .= $strong . '&nbsp;</strong>' . $pm['billing_details']['address']['city']. ', ' .$pm['billing_details']['address']['state'] . ' ' . $pm['billing_details']['address']['postal_code'] . '<br>';
						if (!empty($card_address['country'])) $comment .= $strong . '&nbsp;</strong>' . $pm['billing_details']['address']['country'] . '<br>';
					}
				}
				
				if ($pm['type'] == 'card' && !empty($pm['card'])) {
					$comment .= $hr;
					$card = $pm['card'];
					$comment .= $strong . 'Card Type:</strong>' . (!empty($card['description']) ? $card['description'] : ucwords($card['brand'])) . '<br>';
					$comment .= $strong . 'Card Number:</strong>**** **** **** ' . $card['last4'] . '<br>';
					$comment .= $strong . 'Card Expiry:</strong>' . $card['exp_month'] . ' / ' . $card['exp_year'] . '<br>';
					$comment .= $strong . 'Card Origin:</strong>' . $card['country'] . '<br>';
					$comment .= $hr;
					$comment .= $strong . 'CVC Check:</strong>' . $card['checks']['cvc_check'] . '<br>';
					$comment .= $strong . 'Street Check:</strong>' . $card['checks']['address_line1_check'] . '<br>';
					$comment .= $strong . 'Zip Check:</strong>' . $card['checks']['address_postal_code_check'] . '<br>';
					$comment .= $strong . '3D Secure:</strong>' . (!empty($card['three_d_secure']['result']) ? $card['three_d_secure']['result'] . ' (version ' . $card['three_d_secure']['version'] . ')' : 'not checked') . '<br>';
				}
				
				$this->db->query("INSERT INTO " . DB_PREFIX . "order_history SET order_id = " . (int)$order_id . ", order_status_id = " . (int)$settings['error_status_id'] . ", notify = 0, comment = '" . $this->db->escape($comment) . "', date_added = NOW()");
				
				// Return error
				$this->triggerError('json', $payment_intent['error']['message'], $source_order_id);
				return;
			} elseif ($payment_intent['status'] == 'requires_payment_method' && !in_array($payment_type, array('afterpay_clearpay', 'fpx', 'sepa_debit'))) {
				$this->triggerError('json', 'Missing payment method', $source_order_id);
				return;
			} else {
				$json = array(
					'client_secret'		=> $payment_intent['client_secret'],
					'payment_intent_id'	=> $payment_intent['id'],
					'status'			=> $payment_intent['status'],
				);
			}
		} elseif ($store_card && $stripe_customer_id) {
			// Add payment method to customer
			$attach_response = $this->curlRequest('POST', 'payment_methods/' . $this->request->post['payment_method'] . '/attach', array('customer' => $stripe_customer_id));
			
			if (!empty($attach_response['error']) && !strpos($attach_response['error']['message'], 'already been attached')) {
				$this->triggerError('json', $attach_response['error']['message'], $source_order_id);
				return;
			}
		}
		
		// Set new payment method to default
		if ($store_card && $stripe_customer_id) {
			$customer_data = array(
				'invoice_settings'	=> array(
					'default_payment_method'	=> $this->request->post['payment_method'],
				),
			);
			
			$make_default_response = $this->curlRequest('POST', 'customers/' . $stripe_customer_id, $customer_data);
			
			if (!empty($make_default_response['error'])) {
				$this->triggerError('json', $make_default_response['error']['message'], $source_order_id);
				return;
			}
		}
		
		// Return data
		if ($source) {
			$this->finalizePayment($payment_intent['id'], true, true);
		} else {
			echo json_encode($json);
		}
	}
	
	//==============================================================================
	// finalizePayment()
	//==============================================================================
	public function finalizePayment($payment_intent_id = '', $trigger_subscriptions = true, $ignore_error_handling = false) {
		register_shutdown_function(array($this, 'logFatalErrors'));
		unset($this->session->data[$this->name . '_order_error']);
		
		$settings = $this->getSettings();
		$source_order_id = 0;
		
		// Get order data
		if (!empty($this->session->data['order_id'])) {
			$order_id = $this->session->data['order_id'];
			
			if ($ignore_error_handling) {
				$source_order_id = $order_id;
			}
		} else {
			return;
		}
		
		if (!empty($this->request->post['ignore_error_handling'])) {
			$ignore_error_handling = true;
		}
		
		$this->load->model('checkout/order');
		$order_info = $this->model_checkout_order->getOrder($order_id);
		
		if (isset($this->session->data['stripe_customer_id'])) {
			$stripe_customer_id = $this->session->data['stripe_customer_id'];
			unset($this->session->data['stripe_customer_id']);
		} else {
			$customer_id_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "stripe_customer WHERE customer_id = " . (int)$order_info['customer_id'] . " AND transaction_mode = '" . $this->db->escape($settings['transaction_mode']) . "'");
			$stripe_customer_id = (!empty($customer_id_query->row['stripe_customer_id'])) ? $customer_id_query->row['stripe_customer_id'] : '';
		}
		
		$language = (isset($this->session->data['language'])) ? $this->session->data['language'] : $this->config->get('config_language');
		$currency = $order_info['currency_code'];
		$main_currency = $this->db->query("SELECT * FROM " . DB_PREFIX . "setting WHERE `key` = 'config_currency' AND store_id = 0 ORDER BY setting_id DESC LIMIT 1")->row['value'];
		$decimal_factor = (in_array($settings['currencies_' . $currency], array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF'))) ? 1 : 100;
		
		// Get PaymentIntent ID
		if (isset($this->request->post['payment_intent'])) {
			$payment_intent_id = $this->request->post['payment_intent'];
		}
		
		// Complete order immediately if this is a second payment attempt
		if (strpos($payment_intent_id, 'order_status_id_') === 0) {
			unset($this->session->data[$this->name . '_payment_attempts']);
			unset($this->session->data['src_payment_intent_id']);
			
			if (empty($settings['advanced_error_handling']) || $ignore_error_handling) {
				$this->load->model('checkout/order');
				$this->model_checkout_order->addOrderHistory($order_id, str_replace('order_status_id_', '', $payment_intent_id));
			} else {
				$this->session->data[$this->name . '_order_id'] = $order_id;
				$this->session->data[$this->name . '_order_status_id'] = str_replace('order_status_id_', '', $payment_intent_id);
			}
			
			return;
		}
		
		// Get additional PaymentIntent data
		if ($payment_intent_id) {
			$payment_intent = $this->curlRequest('GET', 'payment_intents/' . $payment_intent_id);
			
			if (!empty($payment_intent['error'])) {
				$this->triggerError('text', $payment_intent['error']['message'], $source_order_id);
				return;
			} else {
				// Re-confirm payment intent if necessary
				if ($payment_intent['status'] == 'requires_confirmation') {
					$confirm_response = $this->curlRequest('POST', 'payment_intents/' . $payment_intent_id . '/confirm');
					
					if (!empty($confirm_response['error'])) {
						$this->triggerError('text', $confirm_response['error']['message'], $source_order_id);
						return;
					} else {
						$payment_intent = $confirm_response;
					}
				}
			}
		}
		
		// Subscribe customer to plans
		$plans = $this->getSubscriptionPlans($settings, $order_info);
		unset($this->session->data[$this->name . '_plans']);
		
		if (!$trigger_subscriptions) {
			foreach ($plans as $plan) {
				$order_info['total'] -= $plan['total_plan_cost'];
			}
		} else {
			// Check for merged subscriptions
			if (!empty($settings['merge_subscriptions']) && $plans) {
				$plan_costs = array();
				$plan_ids = array();
				$plan_names = array();
				$plan_taxed_costs = array();
				$plan_totals = array();
				$plan_items = array();
				
				foreach ($plans as $temp_plan) {
					if (isset($plan_items[$temp_plan['id']])) {
						$plan_items[$temp_plan['id']]['quantity'] += $temp_plan['quantity'];
					} else {
						$plan_costs[] = $temp_plan['cost'];
						$plan_ids[] = $temp_plan['id'];
						$plan_names[] = $temp_plan['name'];
						$plan_taxed_costs[] = $temp_plan['taxed_cost'];
						$plan_totals[] = $temp_plan['total_plan_cost'];
						$plan_items[$temp_plan['id']] = array(
							'plan'		=> $temp_plan['id'],
							'quantity'	=> $temp_plan['quantity'],
							'metadata'	=> array(
								'order_id'		=> $order_id,
								'product_id'	=> $temp_plan['product_id'],
								'product_name'	=> $temp_plan['product_name'],
							),
						);
					}
				}
				
				$combined_plan = $plans[0];
				$combined_plan['cost'] = array_sum($plan_costs);
				$combined_plan['id'] = implode(' + ', $plan_ids);
				$combined_plan['name'] = implode(' + ', $plan_names);
				$combined_plan['quantity'] = 1;
				$combined_plan['taxed_cost'] = array_sum($plan_taxed_costs);
				$combined_plan['total_plan_cost'] = array_sum($plan_totals);
				$combined_plan['items'] = array_values($plan_items);
				
				$plans = array($combined_plan);
			} else {
				foreach ($plans as &$temp_plan) {
					$temp_plan['items'] = array(
						array(
							'plan'		=> $temp_plan['id'],
							'quantity'	=> $temp_plan['quantity'],
							'metadata'	=> array(
								'order_id'		=> $order_id,
								'product_id'	=> $temp_plan['product_id'],
								'product_name'	=> $temp_plan['product_name'],
							),
						),
					);
				}
			}
			
			// Loop through plans
			if ($plans) {
				$order_info['total'] -= $plans[0]['coupon_discount'];
			}
			
			foreach ($plans as &$plan) {
				$subscription_id = '';
				
				// Set up subscription data
				$subscription_data = array(
					'customer'			=> $stripe_customer_id,
					'items'				=> $plan['items'],
					'default_tax_rates'	=> $plan['tax_rates'],
					'metadata'			=> array(
						'order_id'	=> $order_id,
					),
				);
				
				if (empty($settings['merge_subscriptions'])) {
					$subscription_data['metadata']['product_id'] = $plan['product_id'];
					$subscription_data['metadata']['product_name'] = $plan['product_name'];
				}
				
				if (!empty($plan['cycles'])) {
					$subscription_data['metadata']['cycles'] = $plan['cycles'];
				}
				
				if (!empty($plan['coupon_code'])) {
					$subscription_data['coupon'] = $plan['coupon_code'];
				}
				
				if (!empty($plan['shipping_cost'])) {
					// Add temporary trial period
					$subscription_data['trial_period_days'] = ($plan['trial']) ? $plan['trial'] : 1;
					
					$subscription_response = $this->curlRequest('POST', 'subscriptions', $subscription_data);
					
					if (!empty($subscription_response['error'])) {
						$this->triggerError('text', $subscription_response['error']['message'], $source_order_id);
						return;
					}
					
					$subscription_id = $subscription_response['id'];
					
					// Add invoice item for shipping
					$invoice_item_data = array(
						'amount'		=> $this->currency->convert($plan['shipping_cost'] * $decimal_factor, $main_currency, $currency),
						'currency'		=> $settings['currencies_' . $currency],
						'customer'		=> $stripe_customer_id,
						'description'	=> 'Shipping for ' . $plan['name'],
						'subscription'	=> $subscription_id,
					);
					
					$invoice_item_response = $this->curlRequest('POST', 'invoiceitems', $invoice_item_data);
					
					if (!empty($invoice_item_response['error'])) {
						$this->triggerError('text', $invoice_item_response['error']['message'], $source_order_id);
						return;
					}
				}
				
				// Update subscription with real trial period, or start it immediately
				if ($subscription_id) {
					$subscription_data = array();
				}
				
				if (!empty($plan['start_date']) && strtotime($plan['start_date']) > time()) {
					$subscription_data['trial_end'] = strtotime('noon ' . $plan['start_date']);
				} elseif ($plan['trial']) {
					$subscription_data['trial_from_plan'] = 'true';
				} elseif ($subscription_id) {
					$subscription_data['trial_end'] = 'now';
				}
				
				$subscription_response = $this->curlRequest('POST', 'subscriptions' . ($subscription_id ? '/' . $subscription_id : ''), $subscription_data);
				
				if (!empty($subscription_response['error'])) {
					$this->triggerError('text', $subscription_response['error']['message'], $source_order_id);
					return;
				}
				
				// Subtract out subscription costs
				$total_plan_cost = $plan['quantity'] * $plan['taxed_cost'] + $plan['taxed_shipping_cost'];
				$order_info['total'] -= $total_plan_cost;
				
				// Add extra plan data for later use
				$plan['total_plan_cost'] = $total_plan_cost;
				$plan['subscription_response'] = $subscription_response;
			}
		}
		
		// Set initial order_status_id, capture status, and charge data
		if ($settings['charge_mode'] == 'authorize') {
			$capture = false;
			$order_status_id = $settings['authorize_status_id'];
		} else {
			$capture = true;
			$order_status_id = $settings['success_status_id'];
		}
		
		$charge = (!empty($payment_intent['charges']['data'][0])) ? $payment_intent['charges']['data'][0] : array();
		
		// Check fraud data
		if ($settings['charge_mode'] == 'fraud') {
			if (version_compare(VERSION, '2.0.3', '<')) {
				if ($this->config->get('config_fraud_detection')) {
					$this->load->model('checkout/fraud');
					if ($this->model_checkout_fraud->getFraudScore($order_info) > $this->config->get('config_fraud_score')) {
						$capture = false;
						$order_status_id = $settings['authorize_status_id'];
					}
				}
			} else {
				$this->load->model('account/customer');
				$customer_info = $this->model_account_customer->getCustomer($order_info['customer_id']);
				
				if (empty($customer_info['safe'])) {
					$fraud_extensions = $this->db->query("SELECT * FROM " . DB_PREFIX . "extension WHERE `type` = 'fraud' ORDER BY `code` ASC")->rows;
					
					foreach ($fraud_extensions as $extension) {
						$prefix = (version_compare(VERSION, '3.0', '<')) ? '' : 'fraud_';
						if (!$this->config->get($prefix . $extension['code'] . '_status')) continue;
						
						if (version_compare(VERSION, '2.3', '<')) {
							$this->load->model('fraud/' . $extension['code']);
							$fraud_status_id = $this->{'model_fraud_' . $extension['code']}->check($order_info);
						} else {
							$this->load->model('extension/fraud/' . $extension['code']);
							$fraud_status_id = $this->{'model_extension_fraud_' . $extension['code']}->check($order_info);
						}
						
						if ($fraud_status_id) {
							$capture = false;
							$order_status_id = $fraud_status_id;
						}
					}
				}
			}
			
			if (isset($charge['outcome']['type']) && $charge['outcome']['type'] != 'authorized') {
				$capture = false;
				$order_status_id = $settings['authorize_status_id'];
			}
			
			if (isset($charge['outcome']['risk_level']) && $charge['outcome']['risk_level'] == 'highest') {
				$capture = false;
				$order_status_id = $settings['authorize_status_id'];
			}
		}
		
		// Check for address mismatch
		$shipping_address = array(
			'firstname'		=> $order_info['shipping_firstname'],
			'lastname'		=> $order_info['shipping_lastname'],
			'company'		=> $order_info['shipping_company'],
			'address_1'		=> $order_info['shipping_address_1'],
			'address_2'		=> $order_info['shipping_address_2'],
			'city'			=> $order_info['shipping_city'],
			'postcode'		=> $order_info['shipping_postcode'],
			'zone_id'		=> $order_info['shipping_zone_id'],
			'country_id'	=> $order_info['shipping_country_id'],
		);
		
		$payment_address = array(
			'firstname'		=> $order_info['payment_firstname'],
			'lastname'		=> $order_info['payment_lastname'],
			'company'		=> $order_info['payment_company'],
			'address_1'		=> $order_info['payment_address_1'],
			'address_2'		=> $order_info['payment_address_2'],
			'city'			=> $order_info['payment_city'],
			'postcode'		=> $order_info['payment_postcode'],
			'zone_id'		=> $order_info['payment_zone_id'],
			'country_id'	=> $order_info['payment_country_id'],
		);
		
		if (!empty($settings['mismatch_status_id']) && $shipping_address != $payment_address) {
			$order_status_id = $settings['mismatch_status_id'];
			if ($settings['charge_mode'] == 'fraud') {
				$capture = false;
			}
		}
		
		// Capture payment intent if necessary
		if ($order_info['total'] >= 0.5 && $capture && empty($charge['captured']) && $payment_intent_id) {
			/*
			$curl_data = array(
				'amount_to_capture'	=> round($decimal_factor * $this->currency->convert($order_info['total'], $main_currency, $settings['currencies_' . $currency])),
			);
			*/
			
			$capture_response = $this->curlRequest('POST', 'payment_intents/' . $payment_intent_id . '/capture');
			
			if (!empty($capture_response['error'])) {
				$this->triggerError('text', $capture_response['error']['message'], $source_order_id);
				return;
			} else {
				$charge['captured'] = true;
				
				$charge = $this->curlRequest('GET', 'charges/' . $charge['id']);
			}
		}
		
		// Disable logging temporarily, just in case any errors occur that would stop the order from completing
		set_error_handler(function(){});
		
		// Check verifications
		if ($settings['review_status_id'] && isset($charge['outcome']['type']) && $charge['outcome']['type'] != 'authorized')				$order_status_id = $settings['review_status_id'];
		if ($settings['elevated_status_id'] && isset($charge['outcome']['risk_level']) && $charge['outcome']['risk_level'] == 'elevated')	$order_status_id = $settings['elevated_status_id'];
		if ($settings['highest_status_id'] && isset($charge['outcome']['risk_level']) && $charge['outcome']['risk_level'] == 'highest')		$order_status_id = $settings['highest_status_id'];
		
		if (isset($charge['payment_method_details']['card']['checks'])) {
			$checks = $charge['payment_method_details']['card']['checks'];
			if ($settings['street_status_id'] && $checks['address_line1_check'] == 'fail')		$order_status_id = $settings['street_status_id'];
			if ($settings['zip_status_id'] && $checks['address_postal_code_check'] == 'fail')	$order_status_id = $settings['zip_status_id'];
			if ($settings['cvc_status_id'] && $checks['cvc_check'] == 'fail')					$order_status_id = $settings['cvc_status_id'];
		}
		
		if (!empty($charge['billing_details']['address']) && !empty($settings['mismatch_status_id'])) {
			//if ($charge['billing_details']['address']['line1'] != $payment_address['address_1'])		$order_status_id = $settings['mismatch_status_id'];
			if ($charge['billing_details']['address']['city'] != $payment_address['city'])				$order_status_id = $settings['mismatch_status_id'];
			if ($charge['billing_details']['address']['postal_code'] != $payment_address['postcode'])	$order_status_id = $settings['mismatch_status_id'];
		}
		
		// Create comment data
		$strong = '<strong style="display: inline-block; width: 180px; padding: 2px 5px">';
		$hr = '<hr style="margin: 5px">';
		$comment = '';
		
		// Subscription details
		$subscription_response = '';
		
		foreach ($plans as $plan) {
			if (!empty($plan['subscription_response'])) {
				$subscription_response = $plan['subscription_response'];
			}
			
			$comment .= $strong . 'Subscribed to Plan:</strong>' . $plan['name'] . ' (' . $plan['id'] . ')<br>';
			$comment .= $strong . 'Subscription Charge:</strong>' . $this->currency->format($plan['cost'], strtoupper($plan['currency']), 1);
			
			if ($plan['taxed_cost'] != $plan['cost']) {
				$comment .= ' (Including Tax: ' . $this->currency->format($plan['taxed_cost'], strtoupper($plan['currency']), 1) . ')';
			}
			
			if (!empty($plan['shipping_cost'])) {
				$comment .= '<br>' . $strong . 'Shipping Cost:</strong>' . $this->currency->format($plan['shipping_cost'], strtoupper($plan['currency']), 1);
				if ($plan['taxed_shipping_cost'] != $plan['shipping_cost']) {
					$comment .= ' (Including Tax: ' . $this->currency->format($plan['taxed_shipping_cost'], strtoupper($plan['currency']), 1) . ')';
				}
			}
			
			if (!empty($plan['start_date']) && strtotime($plan['start_date']) > time()) {
				$comment .= '<br>' . $strong . 'Start Date:</strong>' . $plan['start_date'];
			} elseif (!empty($plan['trial'])) {
				$comment .= '<br>' . $strong . 'Trial Days:</strong>' . $plan['trial'];
			}
			
			$comment .= $hr;
		}
		
		// Add card details for subscriptions if charge data isn't present
		if (empty($charge) && !empty($subscription_response)) {
			$customer_response = $this->curlRequest('GET', 'customers/' . $subscription_response['customer'], array('expand' => array('invoice_settings.default_payment_method')));
			
			if (!empty($customer_response['invoice_settings']['default_payment_method'])) {
				$pm = $customer_response['invoice_settings']['default_payment_method'];
				
				if (!empty($pm['billing_details'])) {
					$comment .= $strong . 'Billing Details:</strong>' . $pm['billing_details']['name'] . '<br>';
					if (!empty($pm['billing_details']['address'])) {
						$comment .= $strong . '&nbsp;</strong>' . $pm['billing_details']['address']['line1'] . '<br>';
						if (!empty($card_address['line2'])) $comment .= $strong . '&nbsp;</strong>' . $pm['billing_details']['address']['line2'] . '<br>';
						$comment .= $strong . '&nbsp;</strong>' . $pm['billing_details']['address']['city']. ', ' .$pm['billing_details']['address']['state'] . ' ' . $pm['billing_details']['address']['postal_code'] . '<br>';
						if (!empty($card_address['country'])) $comment .= $strong . '&nbsp;</strong>' . $pm['billing_details']['address']['country'] . '<br>';
					}
				}
				
				if ($pm['type'] == 'card') {
					$comment .= $hr;
					$comment .= $strong . 'Card Type:</strong>' . (!empty($pm['card']['description']) ? $pm['card']['description'] : ucwords($pm['card']['brand'])) . '<br>';
					$comment .= $strong . 'Card Number:</strong>**** **** **** ' . $pm['card']['last4'] . '<br>';
					$comment .= $strong . 'Card Expiry:</strong>' . $pm['card']['exp_month'] . ' / ' . $pm['card']['exp_year'] . '<br>';
					$comment .= $strong . 'Card Origin:</strong>' . $pm['card']['country'] . '<br>';
				}
			}
		}
		
		// Charge details
		if (!empty($charge)) {
			$charge_amount = $charge['amount'] / $decimal_factor;
			$comment .= '<script type="text/javascript" src="view/javascript/stripe.js"></script>';
			
			// Get balance_transaction data
			$conversion_and_fee = '';
			$exchange_rate = '';
			
			if (!empty($charge['balance_transaction'])) {
				$balance_transaction = $this->curlRequest('GET', 'balance_transactions/' . $charge['balance_transaction']);
				
				$transaction_currency = strtoupper($balance_transaction['currency']);
				
				if (!empty($settings['currencies_' . $transaction_currency])) {
					$transaction_decimal_factor = (in_array($settings['currencies_' . $transaction_currency], array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF'))) ? 1 : 100;
					
					if (!empty($balance_transaction['exchange_rate'])) {
						$conversion_and_fee .= ' &rarr; ' . $this->currency->format($balance_transaction['amount'] / $transaction_decimal_factor, $transaction_currency, 1);
						$exchange_rate = $strong . 'Exchange Rate:</strong>1.00 ' . strtoupper($charge['currency']) . ' &rarr; ' . ($balance_transaction['exchange_rate']) . ' ' . $transaction_currency . '<br>';
					}
					
					$conversion_and_fee .= ' (Fee: ' . $this->currency->format($balance_transaction['fee'] / $transaction_decimal_factor, $transaction_currency, 1) . ')';
				}
			}
			
			// Universal fields
			$comment .= $strong . 'Stripe Payment ID:</strong><a target="_blank" href="https://dashboard.stripe.com/' . ($settings['transaction_mode'] == 'test' ? 'test/' : '') . 'payments/' . $payment_intent_id . '">' . $payment_intent_id . '</a><br>';
			$comment .= $strong . 'Charge Amount:</strong>' . $this->currency->format($charge_amount, strtoupper($charge['currency']), 1) . $conversion_and_fee . '<br>';
			$comment .= $exchange_rate;
			$comment .= $strong . 'Captured:</strong>' . (!empty($charge['captured']) ? 'Yes' : '<span>No &nbsp;</span> <a onclick="stripeCapture($(this), ' . number_format($charge_amount, 2, '.', '') . ', \'' . $payment_intent_id . '\')">(Capture)</a>') . '<br>';
			
			// Billing details
			if (!empty($charge['billing_details']['name'])) {
				$comment .= $strong . 'Billing Details:</strong>' . $charge['billing_details']['name'] . '<br>';
				if (!empty($charge['billing_details']['address'])) {
					$comment .= $strong . '&nbsp;</strong>' . $charge['billing_details']['address']['line1'] . '<br>';
					if (!empty($card_address['line2'])) $comment .= $strong . '&nbsp;</strong>' . $charge['billing_details']['address']['line2'] . '<br>';
					$comment .= $strong . '&nbsp;</strong>' . $charge['billing_details']['address']['city']. ', ' .$charge['billing_details']['address']['state'] . ' ' . $charge['billing_details']['address']['postal_code'] . '<br>';
					if (!empty($card_address['country'])) $comment .= $strong . '&nbsp;</strong>' . $charge['billing_details']['address']['country'] . '<br>';
				}
				$comment .= $hr;
			}
			
			// Card fields
			if ($charge['payment_method_details']['type'] == 'card') {
				$card = $charge['payment_method_details']['card'];
				
				// Apple Pay fields
				if (!empty($card['wallet']['type']) && $card['wallet']['type'] == 'apple_pay') {
					$comment .= $strong . 'Payment Type:</strong>Apple Pay<br>';
					$comment .= $strong . 'Device Number:</strong>**** **** **** ' . $card['wallet']['dynamic_last4'] . '<br>';
				}
				
				$comment .= $strong . 'Card Type:</strong>' . (!empty($card['description']) ? $card['description'] : ucwords($card['brand'])) . '<br>';
				$comment .= $strong . 'Card Number:</strong>**** **** **** ' . $card['last4'] . '<br>';
				$comment .= $strong . 'Card Expiry:</strong>' . $card['exp_month'] . ' / ' . $card['exp_year'] . '<br>';
				$comment .= $strong . 'Card Origin:</strong>' . $card['country'] . '<br>';
				$comment .= $hr;
				$comment .= $strong . 'CVC Check:</strong>' . $card['checks']['cvc_check'] . '<br>';
				$comment .= $strong . 'Street Check:</strong>' . $card['checks']['address_line1_check'] . '<br>';
				$comment .= $strong . 'Zip Check:</strong>' . $card['checks']['address_postal_code_check'] . '<br>';
				$comment .= $strong . '3D Secure:</strong>' . (!empty($card['three_d_secure']['result']) ? $card['three_d_secure']['result'] . ' (version ' . $card['three_d_secure']['version'] . ')' : 'not checked') . '<br>';
				
				if (!empty($charge['outcome']['risk_level'])) {
					$comment .= $strong . 'Risk Level:</strong>' . $charge['outcome']['risk_level'] . '<br>';
				}
			}
			
			// Non-card payment type
			if (!empty($charge['source']['type'])) {
				$source_type = ($charge['source']['type'] == 'ideal') ? 'iDEAL' : ucwords($charge['source']['type']);
				$comment .= $strong . 'Payment Type:</strong>' . $source_type . '<br>';
			}
			
			// Bancontact fields
			if (isset($charge['source']['bancontact'])) {
				$comment .= $strong . 'Bank Code:</strong>' . $charge['source']['bancontact']['bank_code'] . '<br>';
				$comment .= $strong . 'Bank Name:</strong>' . $charge['source']['bancontact']['bank_name'] . '<br>';
				$comment .= $strong . 'BIC:</strong>' . $charge['source']['bancontact']['bic'] . '<br>';
			}
			
			// FPX fields
			if ($charge['payment_method_details']['type'] == 'fpx') {
				$comment .= $strong . 'Payment Type:</strong>FPX<br>';
				$comment .= $strong . 'Bank Name:</strong>' . $charge['payment_method_details']['fpx']['bank'] . '<br>';
				$comment .= $strong . 'Transaction ID:</strong>' . $charge['payment_method_details']['fpx']['transaction_id'] . '<br>';
			}
			
			// Giropay fields
			if (isset($charge['source']['giropay'])) {
				$comment .= $strong . 'Bank Code:</strong>' . $charge['source']['giropay']['bank_code'] . '<br>';
				$comment .= $strong . 'Bank Name:</strong>' . $charge['source']['giropay']['bank_name'] . '<br>';
				$comment .= $strong . 'BIC:</strong>' . $charge['source']['giropay']['bic'] . '<br>';
			}
			
			// iDEAL fields
			if (isset($charge['source']['ideal'])) {
				$comment .= $strong . 'Bank:</strong>' . $charge['source']['ideal']['bank'] . '<br>';
				$comment .= $strong . 'BIC:</strong>' . $charge['source']['ideal']['bic'] . '<br>';
				$comment .= $strong . 'IBAN:</strong>' . $charge['source']['ideal']['iban_last4'] . '<br>';
			}
			
			// P24 fields
			if (isset($charge['source']['p24'])) {
				$comment .= $strong . 'P24 Reference:</strong>' . $charge['source']['p24']['reference'] . '<br>';
			}
			
			// SEPA fields
			if ($charge['payment_method_details']['type'] == 'sepa_debit') {
				$order_status_id = $settings['initial_status_id'];
				
				$comment .= $strong . 'Payment Type:</strong>SEPA Direct Debit<br>';
				$comment .= $strong . 'Bank Code:</strong>' . $charge['payment_method_details']['sepa_debit']['bank_code'] . '<br>';
				if (!empty($charge['payment_method_details']['sepa_debit']['branch_code'])) {
					$comment .= $strong . 'Branch Code:</strong>' . $charge['payment_method_details']['sepa_debit']['branch_code'] . '<br>';
				}
				$comment .= $strong . 'Bank Account:</strong>' . $charge['payment_method_details']['sepa_debit']['country'] . '** **** **** **** ' . $charge['payment_method_details']['sepa_debit']['last4'] . '<br>';
				$comment .= $strong . 'Fingerprint:</strong>' . $charge['payment_method_details']['sepa_debit']['fingerprint'] . '<br>';
				$comment .= $strong . 'Mandate:</strong>' . $charge['payment_method_details']['sepa_debit']['mandate'] . '<br>';
			}
			
			// Sofort fields
			if (isset($charge['source']['sofort'])) {
				$order_status_id = $settings['initial_status_id'];
				
				$comment .= $strong . 'Bank Code:</strong>' . $charge['source']['sofort']['bank_code'] . '<br>';
				$comment .= $strong . 'Bank Name:</strong>' . $charge['source']['sofort']['bank_name'] . '<br>';
				$comment .= $strong . 'BIC:</strong>' . $charge['source']['sofort']['bic'] . '<br>';
				$comment .= $strong . 'IBAN:</strong>' . $charge['source']['sofort']['iban_last4'] . '<br>';
			}
			
			// Refund link
			$comment .= $hr;
			$comment .= $strong . 'Refund:</strong><a onclick="stripeRefund($(this), ' . number_format($charge_amount, 2, '.', '') . ', \'' . $charge['id'] . '\')">(Refund)</a>';
		}
		
		// Add order history
		$this->db->query("INSERT INTO " . DB_PREFIX . "order_history SET order_id = " . (int)$order_id . ", order_status_id = " . (int)$order_status_id . ", notify = 0, comment = '" . $this->db->escape($comment) . "', date_added = NOW()");
		
		// Subtract trialing subscriptions from order
		$prefix = (version_compare(VERSION, '3.0', '<')) ? '' : 'total_';
		$language_data = $this->load->language(version_compare(VERSION, '2.3', '<') ? 'total/total' : 'extension/total/total');
		
		foreach ($plans as $plan) {
			if ($plan['trial'] || (!empty($plan['start_date']) && strtotime($plan['start_date']) > time())) {
				$this->db->query("UPDATE `" . DB_PREFIX . "order` SET total = " . (float)$order_info['total'] . " WHERE order_id = " . (int)$order_info['order_id']);
				$this->db->query("UPDATE " . DB_PREFIX . "order_total SET value = " . (float)$order_info['total'] . " WHERE order_id = " . (int)$order_info['order_id'] . " AND title = '" . $this->db->escape($language_data['text_total']) . "'");
				$this->db->query("INSERT INTO " . DB_PREFIX . "order_total SET order_id = " . (int)$order_info['order_id'] . ", code = 'total', title = '" . $this->db->escape($settings['text_to_be_charged_' . $language] . ' (' . $plan['name'] . ')') . "', value = " . (float)-$plan['total_plan_cost'] . ", sort_order = " . ((int)$this->config->get($prefix . 'total_sort_order')-1));
			}
		}
		
		// Payment is complete
		restore_error_handler();
		
		unset($this->session->data[$this->name . '_payment_attempts']);
		unset($this->session->data['src_payment_intent_id']);
		
		if (empty($settings['advanced_error_handling']) || $ignore_error_handling) {
			$this->load->model('checkout/order');
			$this->model_checkout_order->addOrderHistory($order_id, $order_status_id);
			return;
		} else {
			$this->session->data[$this->name . '_order_id'] = $order_id;
			$this->session->data[$this->name . '_order_status_id'] = $order_status_id;
		}
		
		// Check 3D Secure for subscriptions
		if (!empty($subscription_response['latest_invoice'])) {
			$invoice_response = $this->curlRequest('GET', 'invoices/' . $subscription_response['latest_invoice']);
			
			if (!empty($invoice_response['payment_intent'])) {
				$payment_intent_response = $this->curlRequest('GET', 'payment_intents/' . $invoice_response['payment_intent']);
				
				if (empty($payment_intent_response['error']) && $payment_intent_response['status'] != 'succeeded') {
					echo $payment_intent_response['client_secret'];
				}
			}
		}
	}
	
	//==============================================================================
	// completeOrder()
	//==============================================================================
	public function completeOrder() {
		if (empty($this->session->data[$this->name . '_order_id'])) {
			echo 'No order data';
			return;
		}
		
		$order_id = $this->session->data[$this->name . '_order_id'];
		$order_status_id = $this->session->data[$this->name . '_order_status_id'];
		
		unset($this->session->data[$this->name . '_order_id']);
		unset($this->session->data[$this->name . '_order_status_id']);
		
		$this->session->data[$this->name . '_order_error'] = $order_id;
		
		$this->load->model('checkout/order');
		$this->model_checkout_order->addOrderHistory($order_id, $order_status_id);
	}
	
	//==============================================================================
	// completeWithError()
	//==============================================================================
	public function completeWithError() {
		if (empty($this->session->data[$this->name . '_order_error'])) {
			echo 'Payment was not processed';
			return;
		}
		
		$settings = $this->getSettings();
		
		$this->db->query("UPDATE `" . DB_PREFIX . "order` SET order_status_id = " . (int)$settings['error_status_id'] . ", date_modified = NOW() WHERE order_id = " . (int)$this->session->data[$this->name . '_order_error']);
		$this->db->query("INSERT INTO " . DB_PREFIX . "order_history SET order_id = " . (int)$this->session->data[$this->name . '_order_error'] . ", order_status_id = " . (int)$settings['error_status_id'] . ", notify = 0, comment = 'The order could not be completed normally due to the following error:<br><br><em>" . $this->db->escape($this->request->post['error_message']) . "</em><br><br>Double-check your SMTP settings in System > Settings > Mail, and then try disabling or uninstalling any modifications that affect customer orders (i.e. the /catalog/model/checkout/order.php file). One of those is usually the cause of errors like this.', date_added = NOW()");
		
		unset($this->session->data[$this->name . '_order_error']);
	}
	
	//==============================================================================
	// checkoutComplete()
	//==============================================================================
	public function checkoutComplete() {
		if (empty($this->session->data['stripe_checkout_session_id'])) {
			echo 'No checkout session ID';
			return;
		} else {
			$session_id = $this->session->data['stripe_checkout_session_id'];
			unset($this->session->data['stripe_checkout_session_id']);
		}
		
		$checkout_session = $this->curlRequest('GET', 'checkout/sessions/' . $session_id);
		
		if (!empty($checkout_session['error'])) {
			echo $checkout_session['error']['message'];
			return;
		}
		
		if (!empty($checkout_session['metadata']['quick_buy'])) {
			$payment_intent = $this->curlRequest('GET', 'payment_intents/' . $checkout_session['payment_intent']);
			
			if (!empty($payment_intent['error']) || empty($payment_intent['charges']['data'][0])) {
				$this->log->write('STRIPE PAYMENT GATEWAY: No charge data in Checkout Session for order ' . $checkout_session['metadata']['order_id']);
			} else {
				$charge = $payment_intent['charges']['data'][0];
				
				// Set up name and address data
				$names = explode(' ', $charge['billing_details']['name'], 2);
				$firstname = $names[0];
				$lastname = (isset($names[1])) ? $names[1] : '';
				
				$country_id = 0;
				$country_name = $charge['billing_details']['address']['country'];
				$country_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "country WHERE iso_code_2 = '" . $this->db->escape($charge['billing_details']['address']['country']) . "'");
				if ($country_query->num_rows) {
					$country_id = $country_query->row['country_id'];
					$country_name = $country_query->row['name'];
				}
				
				$zone_id = 0;
				$zone_name = $charge['billing_details']['address']['state'];
				$zone_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "zone WHERE `name` = '" . $this->db->escape($charge['billing_details']['address']['state']) . "' AND country_id = " . (int)$country_id);
				if ($zone_query->num_rows) {
					$zone_id = $zone_query->row['zone_id'];
				} else {
					$zone_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "zone WHERE `code` = '" . $this->db->escape($charge['billing_details']['address']['state']) . "' AND country_id = " . (int)$country_id);
					if ($zone_query->num_rows) {
						$zone_id = $zone_query->row['zone_id'];
						$zone_name = $zone_query->row['name'];
					}
				}
				
				// Update order in OpenCart
				$this->db->query("
					UPDATE `" . DB_PREFIX . "order` SET
					firstname = '" . $this->db->escape($firstname) . "',
					lastname = '" . $this->db->escape($lastname) . "',
					email = '" . $this->db->escape($checkout_session['customer_details']['email']) . "',
					telephone = '" . $this->db->escape($checkout_session['customer_details']['phone']) . "',
					payment_firstname = '" . $this->db->escape($firstname) . "',
					payment_lastname = '" . $this->db->escape($lastname) . "',
					payment_address_1 = '" . $this->db->escape($charge['billing_details']['address']['line1']) . "',
					payment_address_2 = '" . $this->db->escape($charge['billing_details']['address']['line2']) . "',
					payment_city = '" . $this->db->escape($charge['billing_details']['address']['city']) . "',
					payment_postcode = '" . $this->db->escape($charge['billing_details']['address']['postal_code']) . "',
					payment_country = '" . $this->db->escape($country_name) . "',
					payment_country_id = " . (int)$country_id . ",
					payment_zone = '" . $this->db->escape($zone_name) . "',
					payment_zone_id = " . (int)$zone_id . "
					WHERE order_id = " . (int)$checkout_session['metadata']['order_id']
				);
				
				// Updated PaymentIntent in Stripe
				$this->load->model('checkout/order');
				$order_info = $this->model_checkout_order->getOrder($checkout_session['metadata']['order_id']);
				
				$settings = $this->getSettings();
				$description = $this->replaceShortcodes($settings['transaction_description'], $order_info);
				$customer_info = $order_info['firstname'] . ' ' . $order_info['lastname'] . ', ' . $order_info['email'] . ', ' . $order_info['telephone'] . ', customer_id: ' . $order_info['customer_id'];
				
				$this->curlRequest('POST', 'payment_intents/' . $checkout_session['payment_intent'], array('description' => $description, 'metadata' => array('Customer Info' => $customer_info)));
			}
			
			$this->request->get['payment_method'] = 'Quick Buy Button';
		}
		
		$this->completeImmediately();
	}
	
	//==============================================================================
	// completeImmediately()
	//==============================================================================
	public function completeImmediately() {
		if (!empty($this->request->get['payment_method']) && in_array($this->request->get['payment_method'], array('afterpay', 'fpx'))) {
			if (empty($this->request->get['payment_intent'])) {
				$this->request->get['redirect_status'] = 'Missing payment_intent for return URL.';
			} else {
				$payment_intent_response = $this->curlRequest('GET', 'payment_intents/' . $this->request->get['payment_intent']);
				
				if ($payment_intent_response['status'] != 'succeeded') {
					$this->request->get['redirect_status'] = 'Payment failed. Please try another payment method.';
				} elseif ($this->request->get['payment_method'] == 'fpx') {
					$this->session->data['order_id'] = $payment_intent_response['metadata']['Order ID'];
					$this->finalizePayment($payment_intent_response['id'], false, true);
				}
			}
		}
		
		if (!empty($this->request->get['redirect_status']) && $this->request->get['redirect_status'] != 'pending' && $this->request->get['redirect_status'] != 'succeeded') {
			$this->errorPage($this->request->get['redirect_status']);
			return;
		}
		
		if (!empty($this->session->data['stripe_checkout_session_id'])) {
			$this->errorPage('Checkout session ID present');
			unset($this->session->data['stripe_checkout_session_id']);
			return;
		}
		
		if (!empty($this->session->data['order_id'])) {
			$order_id = $this->session->data['order_id'];
			$order_info = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE order_id = " . (int)$order_id)->row;
			
			if (empty($order_info['order_status_id'])) {
				$settings = $this->getSettings();
				$comment = '';
				
				if (!empty($this->request->get['payment_method'])) {
					$comment = 'Payment Method: ' . ucwords($this->request->get['payment_method']);
				}
				
				$this->load->model('checkout/order');
				$this->model_checkout_order->addOrderHistory($order_id, $settings['initial_status_id'], $comment);
			}
		}
		
		$this->response->redirect($this->url->link('checkout/success', '', 'SSL'));
	}
	
	//==============================================================================
	// Webhook functions
	//==============================================================================
	public function webhook() {
		register_shutdown_function(array($this, 'logFatalErrors'));
		$settings = $this->getSettings();
		$language = $this->config->get('config_language');

		$event = @json_decode(file_get_contents('php://input'), true);
		
		if (empty($event['type'])) {
			echo 'Stripe Payment Gateway webhook is working.';
			return;
		}
		
		if (!isset($this->request->get['key']) || $this->request->get['key'] != md5($this->config->get('config_encryption'))) {
			echo 'Wrong key';
			$this->log->write('STRIPE WEBHOOK ERROR: webhook URL key ' . $this->request->get['key'] . ' does not match the encryption key hash ' . md5($this->config->get('config_encryption')));
			return;
		}
		
		// Register successful webhook call
		echo 'success';
		
		$webhook = $event['data']['object'];
		$this->load->model('checkout/order');
		
		if ($event['type'] == 'customer.deleted') {
			
			$mode = ($webhook['livemode']) ? 'live' : 'test';
			$this->db->query("DELETE FROM " . DB_PREFIX . "stripe_customer WHERE stripe_customer_id = '" . $this->db->escape($webhook['id']) . "' AND transaction_mode = '" . $this->db->escape($mode) . "'");
			
		} elseif ($event['type'] == 'charge.captured') {
			
			if ($settings['charge_mode'] != 'authorize') return;
			
			$order_history_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_history WHERE `comment` LIKE '%" . $this->db->escape($webhook['id']) . "%' ORDER BY order_history_id DESC");
			if (!$order_history_query->num_rows) return;
			
			$strong = '<strong style="display: inline-block; width: 140px; padding: 3px">';
			$comment = $strong . 'Stripe Event:</strong>' . $event['type'] . '<br>';
			
			$order_id = $order_history_query->row['order_id'];
			$order_status_id = ($settings['success_status_id']) ? $settings['success_status_id'] : $order_history_query->row['order_status_id'];
			
			$this->model_checkout_order->addOrderHistory($order_id, $order_status_id, $comment, false);
			
		} elseif ($event['type'] == 'charge.refunded') {
			
			if (empty($webhook['payment_intent'])) return;
			
			$order_history_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_history WHERE `comment` LIKE '%" . $this->db->escape($webhook['payment_intent']) . "%' ORDER BY order_history_id DESC");
			if (!$order_history_query->num_rows) return;
			
			$refund = array_pop($webhook['refunds']['data']);
			$refund_currency = strtoupper($refund['currency']);
			$decimal_factor = (in_array($refund_currency, array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF'))) ? 1 : 100;
			
			$strong = '<strong style="display: inline-block; width: 170px; padding: 3px">';
			$comment = $strong . 'Stripe Event:</strong>' . $event['type'] . '<br>';
			$comment .= $strong . 'Refund ID:</strong>' . $refund['id'] . '<br>';
			$comment .= $strong . 'Refund Amount:</strong>' . $this->currency->format($refund['amount'] / $decimal_factor, $refund_currency, 1) . '<br>';
			$comment .= $strong . 'Total Amount Refunded:</strong>' . $this->currency->format($webhook['amount_refunded'] / $decimal_factor, $refund_currency, 1);
			
			$order_id = $order_history_query->row['order_id'];
			$order_info = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE order_id = " . (int)$order_id)->row;
			$refund_type = ($webhook['amount_refunded'] == $webhook['amount']) ? 'refund' : 'partial';
			$order_status_id = ($settings[$refund_type . '_status_id']) ? $settings[$refund_type . '_status_id'] : $order_info['order_status_id'];
			
			$this->model_checkout_order->addOrderHistory($order_id, $order_status_id, $comment, false);
		
		} elseif ($event['type'] == 'charge.failed') {
			
			if ($webhook['payment_method_details']['type'] != 'sofort') return;
			
			$order_id = $webhook['metadata']['Order ID'];
			$order_info = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE order_id = " . (int)$order_id)->row;
			
			if ($order_info['order_status_id'] == $settings['initial_status_id'] && $settings['error_status_id']) {
				$this->model_checkout_order->addOrderHistory($order_id, $settings['error_status_id'], '<b>PAYMENT FAILED<br><br>Error:</b> ' . $webhook['failure_message'], false);
			}
			
			if (!empty($settings['delayed_payment_emails'])) {
				$admin_emails = explode(',', $settings['delayed_payment_emails']);
				$subject = 'Payment Failed For Order #' . $order_id;
				$message = 'The payment for order #' . $order_id . ' has failed with the error message "' . $webhook['failure_message'] . '"';
				$this->sendEmail($admin_emails, $subject, $message);
			}
			
		} elseif ($event['type'] == 'charge.succeeded') {
			
			if ($webhook['payment_method_details']['type'] != 'sofort') return;
			
			$order_id = $webhook['metadata']['Order ID'];
			$order_info = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE order_id = " . (int)$order_id)->row;
			$order_status_id = ($order_info['order_status_id'] == $settings['initial_status_id']) ? $settings['success_status_id'] : $order_info['order_status_id'];
			
			$this->model_checkout_order->addOrderHistory($order_id, $order_status_id, '<b>PAYMENT SUCCEEDED<br><br>Stripe Charge ID:</b> ' . $webhook['id'], false);
			
			if (!empty($settings['delayed_payment_emails'])) {
				$admin_emails = explode(',', $settings['delayed_payment_emails']);
				$subject = 'Payment Completed For Order #' . $order_id;
				$message = 'The payment for order #' . $order_id . ' has been completed successfully.';
				$this->sendEmail($admin_emails, $subject, $message);
			}
			
		} elseif ($event['type'] == 'checkout.session.completed') {
			
			$order_info = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE order_id = " . (int)$webhook['metadata']['order_id'])->row;
			$explode = explode('/', $order_info['store_url']);
			$store_domain = $explode[2];
			
			if ($this->request->server['HTTP_HOST'] == $store_domain || strpos($this->request->server['HTTP_HOST'], 'ngrok')) {
				$this->session->data['order_id'] = $webhook['metadata']['order_id'];
				$this->finalizePayment($webhook['payment_intent'], false, true);
			}
			
		} elseif ($event['type'] == 'customer.subscription.deleted') {
			
			$customer_response = $this->curlRequest('GET', 'customers/' . $webhook['customer']);
			
			if (empty($customer_response['error'])) {
				$subject = 'Canceled Subscription for Order #' . $webhook['metadata']['order_id'];
				$product_name = (!empty($webhook['price']['metadata']['product_name'])) ? $webhook['price']['metadata']['product_name'] : $webhook['metadata']['product_name'];
				$message = 'Subscription: ' . $product_name . ' (' . $webhook['id'] . ')<br>Customer: ' . $customer_response['description'] . ' ' . $customer_response['email'];
				$this->sendEmail($this->config->get('config_email'), $subject, $message);
			}
			
		} elseif ($event['type'] == 'payment_intent.payment_failed') {
			
			if (empty($webhook['charges']['data'][0]['payment_method_details']['type'])) return;
			
			$valid_payment_types = array('afterpay_clearpay', 'sepa_debit');
			if (!in_array($webhook['charges']['data'][0]['payment_method_details']['type'], $valid_payment_types)) return;
			
			$order_id = $webhook['metadata']['Order ID'];
			$order_info = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE order_id = " . (int)$order_id)->row;
			
			if ($order_info['order_status_id'] == $settings['initial_status_id'] && $settings['error_status_id']) {
				$this->model_checkout_order->addOrderHistory($order_id, $settings['error_status_id'], '<b>PAYMENT FAILED<br><br>Error:</b> ' . $webhook['last_payment_error']['message'], false);
			}
			
			if (!empty($settings['delayed_payment_emails'])) {
				$admin_emails = explode(',', $settings['delayed_payment_emails']);
				$subject = 'Payment Failed For Order #' . $order_id;
				$message = 'The payment for order #' . $order_id . ' has failed with the error message "' . $webhook['last_payment_error']['message'] . '"';
				$this->sendEmail($admin_emails, $subject, $message);
			}
			
		} elseif ($event['type'] == 'payment_intent.succeeded') {
			
			if (empty($webhook['charges']['data'][0]['payment_method_details']['type'])) return;
			
			$valid_payment_types = array('afterpay_clearpay', 'sepa_debit');
			if (!in_array($webhook['charges']['data'][0]['payment_method_details']['type'], $valid_payment_types)) return;
			
			$order_id = $webhook['metadata']['Order ID'];
			$order_info = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE order_id = " . (int)$order_id)->row;
			$order_status_id = ($order_info['order_status_id'] == $settings['initial_status_id']) ? $settings['success_status_id'] : $order_info['order_status_id'];
			$charge_id = (!empty($webhook['charges']['data'][0]['id'])) ? $webhook['charges']['data'][0]['id'] : '';
			
			$this->model_checkout_order->addOrderHistory($order_id, $order_status_id, '<b>PAYMENT SUCCEEDED<br><br>Stripe Charge ID:</b> ' . $charge_id, false);
			
			if (!empty($settings['delayed_payment_emails'])) {
				$admin_emails = explode(',', $settings['delayed_payment_emails']);
				$subject = 'Payment Completed For Order #' . $order_id;
				$message = 'The payment for order #' . $order_id . ' has been completed successfully.';
				$this->sendEmail($admin_emails, $subject, $message);
			}
			
		} elseif ($event['type'] == 'source.canceled') {
			
			if (!empty($settings['delayed_payment_emails']) && !empty($webhook['metadata']['order_id'])) {
				$order_id = $webhook['metadata']['order_id'];
				$admin_emails = explode(',', $settings['delayed_payment_emails']);
				$subject = 'Payment Canceled For Order #' . $order_id;
				$message = 'The payment for order #' . $order_id . ' has been canceled without completing.';
				$this->sendEmail($admin_emails, $subject, $message);
			}
			
		} elseif ($event['type'] == 'source.chargeable') {
			
			if (!empty($webhook['metadata']['order_id'])) {
				$order_id = $webhook['metadata']['order_id'];
				$this->session->data['order_id'] = $order_id;
				$this->createPaymentIntent($webhook);
			}
			
		} elseif ($event['type'] == 'invoice.payment_succeeded' && !empty($settings['subscriptions'])) {
			
			// Check for duplicate webhook
			$event_id_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_history WHERE `comment` LIKE '%" . $this->db->escape($event['id']) . "%'");
			if ($event_id_query->num_rows) {
				return;
			}
			
			// Check for 0.00 trial invoices
			if (empty($webhook['total'])) {
				return;
			}
			
			// Set customer data
			$data = array();
			$data['email'] = $webhook['customer_email'];
			
			$opencart_customer = $this->db->query("SELECT * FROM " . DB_PREFIX . "customer WHERE email = '" . $this->db->escape($data['email']) . "'")->row;
			$data['customer_id'] = (!empty($opencart_customer['customer_id'])) ? $opencart_customer['customer_id'] : 0;
			
			// Set customer name and telephone
			$customer_response = $this->curlRequest('GET', 'customers/' . $webhook['customer'], array('expand' => array('default_source')));
			$stripe_customer = (!empty($customer_response['error'])) ? $customer_response['default_source']['owner'] : array();
			
			if (!empty($webhook['customer_name'])) {
				$customer_name = explode(' ', $webhook['customer_name'], 2);
			} elseif (!empty($stripe_customer['name'])) {
				$customer_name = explode(' ', $stripe_customer['name'], 2);
			} elseif (!empty($opencart_customer['firstname'])) {
				$customer_name = array($opencart_customer['firstname'], $opencart_customer['lastname']);
			}
			
			$data['firstname'] = (isset($customer_name[0])) ? $customer_name[0] : '';
			$data['lastname'] = (isset($customer_name[1])) ? $customer_name[1] : '';
			
			if (!empty($webhook['customer_phone'])) {
				$data['telephone'] = $webhook['customer_phone'];
			} elseif (!empty($stripe_customer['phone'])) {
				$data['telephone'] = $stripe_customer['phone'];
			} elseif (!empty($opencart_customer['telephone'])) {
				$data['telephone'] = $opencart_customer['telephone'];
			} else {
				$data['telephone'] = '';
			}
			
			// Set billing address
			if (!empty($webhook['customer_address'])) {
				$billing_address = $webhook['customer_address'];
			} elseif (!empty($stripe_customer['address'])) {
				$billing_address = $stripe_customer['address'];
			} else {
				$billing_address = array(
					'line1'			=> '',
					'line2'			=> '',
					'city'			=> '',
					'state'			=> '',
					'postal_code'	=> '',
					'country'		=> '',
				);
			}
			
			$country_id = 0;
			$country_name = $billing_address['country'];
			$country_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "country WHERE iso_code_2 = '" . $this->db->escape($billing_address['country']) . "'");
			if ($country_query->num_rows) {
				$country_id = $country_query->row['country_id'];
				$country_name = $country_query->row['name'];
			}
			
			$zone_id = 0;
			$zone_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "zone WHERE `name` = '" . $this->db->escape($billing_address['state']) . "' AND country_id = " . (int)$country_id);
			if ($zone_query->num_rows) {
				$zone_id = $zone_query->row['zone_id'];
			} else {
				$zone_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "zone WHERE `code` = '" . $this->db->escape($billing_address['state']) . "' AND country_id = " . (int)$country_id);
				if ($zone_query->num_rows) {
					$zone_id = $zone_query->row['zone_id'];
				}
			}
			
			$data['payment_firstname']	= $data['firstname'];
			$data['payment_lastname']	= $data['lastname'];
			$data['payment_company']	= '';
			$data['payment_company_id']	= '';
			$data['payment_tax_id']		= '';
			$data['payment_address_1']	= $billing_address['line1'];
			$data['payment_address_2']	= $billing_address['line2'];
			$data['payment_city']		= $billing_address['city'];
			$data['payment_postcode']	= $billing_address['postal_code'];
			$data['payment_zone_id']	= $zone_id;
			$data['payment_zone']		= $billing_address['state'];
			$data['payment_country_id']	= $country_id;
			$data['payment_country']	= $country_name;
			
			// Set shipping address
			if ($settings['order_address'] == 'stripe') {
				if (!empty($webhook['customer_shipping'])) {
					$shipping_name = explode(' ', $webhook['customer_shipping']['name'], 2);
					$shipping_address = $webhook['customer_shipping']['address'];
					
					$country_id = 0;
					$country_name = $shipping_address['country'];
					$country_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "country WHERE iso_code_2 = '" . $this->db->escape($shipping_address['country']) . "'");
					if ($country_query->num_rows) {
						$country_id = $country_query->row['country_id'];
						$country_name = $country_query->row['name'];
					}
					
					$zone_id = 0;
					$zone_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "zone WHERE `name` = '" . $this->db->escape($shipping_address['state']) . "' AND country_id = " . (int)$country_id);
					if ($zone_query->num_rows) {
						$zone_id = $zone_query->row['zone_id'];
					} else {
						$zone_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "zone WHERE `code` = '" . $this->db->escape($shipping_address['state']) . "' AND country_id = " . (int)$country_id);
						if ($zone_query->num_rows) {
							$zone_id = $zone_query->row['zone_id'];
						}
					}
					
					$data['shipping_firstname']		= $shipping_name[0];
					$data['shipping_lastname']		= (isset($shipping_name[1]) ? $shipping_name[1] : '');
					$data['shipping_company']		= '';
					$data['shipping_company_id']	= '';
					$data['shipping_tax_id']		= '';
					$data['shipping_address_1']		= $shipping_address['line1'];
					$data['shipping_address_2']		= $shipping_address['line2'];
					$data['shipping_city']			= $shipping_address['city'];
					$data['shipping_postcode']		= $shipping_address['postal_code'];
					$data['shipping_zone_id']		= $zone_id;
					$data['shipping_zone']			= $shipping_address['state'];
					$data['shipping_country_id']	= $country_id;
					$data['shipping_country']		= $country_name;
				} else {
					foreach (array('firstname', 'lastname', 'company', 'company_id', 'tax_id', 'address_1', 'address_2', 'city', 'postcode', 'zone_id', 'zone', 'country_id', 'country') as $field) {
						$data['shipping_' . $field] = $data['payment_' . $field];
					}
				}
			} else {
				if ($settings['order_address'] == 'opencart' && !empty($opencart_customer)) {
					if (!empty($opencart_customer['address_id'])) {
						$opencart_address = $this->db->query("SELECT * FROM " . DB_PREFIX . "address WHERE address_id = " . (int)$opencart_customer['address_id'])->row;
					} else {
						$opencart_address = $this->db->query("SELECT * FROM " . DB_PREFIX . "address WHERE customer_id = " . (int)$opencart_customer['customer_id'] . " ORDER BY address_id DESC")->row;
					}
				}
				
				if ($settings['order_address'] == 'original' || empty($opencart_address)) {
					$original_order_id = 0;
					
					foreach ($webhook['lines']['data'] as $line) {
						if (!empty($line['metadata']['order_id'])) {
							$original_order_id = $line['metadata']['order_id'];
						}
					}
					
					$order_info = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE order_id = " . (int)$original_order_id)->row;
					
					if (!empty($order_info['shipping_firstname'])) {
						$opencart_address = array(
							'customer_id'	=> $order_info['customer_id'],
							'firstname'		=> $order_info['shipping_firstname'],
							'lastname'		=> $order_info['shipping_lastname'],
							'company'		=> $order_info['shipping_company'],
							'address_1'		=> $order_info['shipping_address_1'],
							'address_2'		=> $order_info['shipping_address_2'],
							'city'			=> $order_info['shipping_city'],
							'postcode'		=> $order_info['shipping_postcode'],
							'country_id'	=> $order_info['shipping_country_id'],
							'zone_id'		=> $order_info['shipping_zone_id'],
							'custom_field'	=> (!empty($order_info['shipping_custom_field'])) ? $order_info['shipping_custom_field'] : '',
						);
					}
				}
				
				$zone_id = (isset($opencart_address['zone_id'])) ? $opencart_address['zone_id'] : 0;
				$zone_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "zone WHERE zone_id = " . (int)$opencart_address['zone_id']);
				$opencart_address['zone'] = (isset($zone_query->row['name'])) ? $zone_query->row['name'] : '';
				
				$country_id = (isset($opencart_address['country_id'])) ? $opencart_address['country_id'] : 0;
				$country_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "country WHERE country_id = " . (int)$opencart_address['country_id']);
				$opencart_address['country'] = (isset($country_query->row['name'])) ? $country_query->row['name'] : '';
				
				foreach (array('firstname', 'lastname', 'company', 'company_id', 'tax_id', 'address_1', 'address_2', 'city', 'postcode', 'zone_id', 'zone', 'country_id', 'country') as $field) {
					$data['shipping_' . $field] = (isset($opencart_address[$field])) ? $opencart_address[$field] : '';
				}
			}
			
			// Set products and line items
			$data['payment_method']		= html_entity_decode($settings['title_' . $language], ENT_QUOTES, 'UTF-8');
			$data['payment_code']		= $this->name;
			$data['shipping_method']	= '(none)';
			$data['shipping_code']		= '(none)';
			
			$original_order_id = 0;
			$plan_ids = array();
			$product_data = array();
			$shipping_amount = 0;
			$subtotal = 0;
			$total_data = array();
			
			foreach ($webhook['lines']['data'] as $line) {
				// Find original order_id
				if (!empty($line['metadata']['order_id'])) {
					$original_order_id = $line['metadata']['order_id'];
					$original_order_query = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE order_id = " . (int)$original_order_id);
					
					if ($original_order_query->num_rows) {
						$data['payment_method'] = $original_order_query->row['payment_method'];
						$data['shipping_method'] = $original_order_query->row['shipping_method'];
						$data['shipping_code'] = $original_order_query->row['shipping_code'];
					}
				}
				
				// Decrement cycles if set
				if ($line['type'] == 'subscription' && !empty($line['metadata']['cycles'])) {
					if ($line['metadata']['cycles'] == 1) {
						$this->curlRequest('DELETE', 'subscriptions/' . $line['subscription']);
						// or to avoid prorating, use the following line instead
						//$this->curlRequest('POST', 'subscriptions/' . $line['subscription'], array('cancel_at_period_end' => true));
					} else {
						$line['metadata']['cycles'] -= 1;
						$this->curlRequest('POST', 'subscriptions/' . $line['subscription'], array('metadata' => $line['metadata']));
					}
				}
				
				// Add line item to order
				$line_currency = strtoupper($line['currency']);
				$line_decimal_factor = (in_array($line_currency, array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF'))) ? 1 : 100;
				
				if (empty($line['plan'])) {
					
					$shipping_line_item = (strpos($line['description'], 'Shipping for') === 0);
					
					// Add non-product line items
					$total_data[] = array(
						'code'			=> ($shipping_line_item) ? 'shipping' : 'total',
						'title'			=> $line['description'],
						'text'			=> $this->currency->format($line['amount'] / $line_decimal_factor, $line_currency, 1),
						'value'			=> $line['amount'] / $line_decimal_factor,
						'sort_order'	=> 2,
					);
					
					// Add invoice item for shipping
					if ($shipping_line_item) {
						$shipping_amount = $line['amount'] / $line_decimal_factor;
						
						if ($data['shipping_method'] == '(none)') {
							$data['shipping_method'] = $line['description'];
						}
						
						$invoice_item_data = array(
							'amount'		=> $line['amount'],
							'currency'		=> $line['currency'],
							'customer'		=> $webhook['customer'],
							'description'	=> $line['description'],
							'subscription'	=> $line['subscription'],
						);
						
						$invoice_item_response = $this->curlRequest('POST', 'invoiceitems', $invoice_item_data);
						
						if (!empty($invoice_item_response['error'])) {
							$this->log->write('STRIPE WEBHOOK ERROR: ' . $invoice_item_response['error']['message']);
						}
					}
					
				} else {
					
					// Add product corresponding to line item
					if (!empty($line['price']['metadata']['product_id'])) {
						$product_id = $line['price']['metadata']['product_id'];
						$product_name = $line['price']['metadata']['product_name'];
					} elseif (!empty($line['metadata']['product_id'])) {
						$product_id = $line['metadata']['product_id'];
						$product_name = $line['metadata']['product_name'];
					} else {
						$product_id = 0;
						$product_name = $line['description'];
					}
					
					$plan_ids[] = $line['plan']['id'];
					$charge = $line['amount'] / $line_decimal_factor;
					$subtotal += $charge;
					
					if (!empty($product_id)) {
						$product_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "product p LEFT JOIN " . DB_PREFIX . "product_description pd ON (p.product_id = pd.product_id AND pd.language_id = " . (int)$this->config->get('config_language_id') . ") WHERE p.product_id = " . (int)$product_id);
					} else {
						$product_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "product p LEFT JOIN " . DB_PREFIX . "product_description pd ON (p.product_id = pd.product_id AND pd.language_id = " . (int)$this->config->get('config_language_id') . ") WHERE p.location = '" . $this->db->escape($line['plan']['id']) . "'");
					}
					
					if ($product_query->num_rows) {
						$product = $product_query->row;
						$product['name'] = $product_name;
					} else {
						$product = array(
							'product_id'	=> 0,
							'name'			=> $product_name,
							'model'			=> '',
							'subtract'		=> 0,
							'tax_class_id'	=> 0,
							'shipping'		=> 1,
						);
					}
					
					$product_data[] = array(
						'product_id'	=> $product['product_id'],
						'name'			=> $product['name'],
						'model'			=> $product['model'],
						'option'		=> array(),
						'download'		=> array(),
						'quantity'		=> $line['quantity'],
						'subtract'		=> $product['subtract'],
						'price'			=> ($charge / $line['quantity']),
						'total'			=> $charge,
						'tax'			=> $this->tax->getTax($charge, $product['tax_class_id']),
						'reward'		=> isset($product['reward']) ? $product['reward'] : 0
					);
				}
				
			}
			
			// Set order totals
			$data['currency_code'] = strtoupper($webhook['currency']);
			$data['currency_id'] = $this->currency->getId($data['currency_code']);
			$data['currency_value'] = $this->currency->getValue($data['currency_code']);
			
			$decimal_factor = (in_array($data['currency_code'], array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF'))) ? 1 : 100;
			
			$total_data[] = array(
				'code'			=> 'sub_total',
				'title'			=> 'Sub-Total',
				'text'			=> $this->currency->format($subtotal, $data['currency_code'], 1),
				'value'			=> $subtotal,
				'sort_order'	=> 1,
			);
			
			if (!empty($webhook['discount']['coupon'])) {
				if (!empty($webhook['discount']['coupon']['amount_off'])) {
					$discount_amount = $webhook['discount']['coupon']['amount_off'] / $decimal_factor;
				} else {
					$discount_amount = ($subtotal + $shipping_amount) * $webhook['discount']['coupon']['percent_off'] / 100;
				}
				
				$total_data[] = array(
					'code'			=> 'coupon',
					'title'			=> $webhook['discount']['coupon']['name'] . ' (' . $webhook['discount']['coupon']['id'] . ')',
					'text'			=> $this->currency->format(-$discount_amount, $data['currency_code'], 1),
					'value'			=> -$discount_amount,
					'sort_order'	=> 3,
				);
			}
			
			if (!empty($webhook['tax'])) {
				$total_data[] = array(
					'code'			=> 'tax',
					'title'			=> 'Tax',
					'text'			=> $this->currency->format($webhook['tax'] / $decimal_factor, $data['currency_code'], 1),
					'value'			=> $webhook['tax'] / $decimal_factor,
					'sort_order'	=> 4,
				);
			}
			
			$total_data[] = array(
				'code'			=> 'total',
				'title'			=> 'Total',
				'text'			=> $this->currency->format($webhook['total'] / $decimal_factor, $data['currency_code'], 1),
				'value'			=> $webhook['total'] / $decimal_factor,
				'sort_order'	=> 5,
			);
			
			$data['products'] = $product_data;
			$data['totals'] = $total_data;
			$data['total'] = $webhook['total'] / $decimal_factor;
			
			// Check for immediate subscriptions
			$now_query = $this->db->query("SELECT NOW()");
			
			if (!empty($original_order_query->row)) {
				if ((strtotime($now_query->row['NOW()']) - strtotime($original_order_query->row['date_added'])) < 82800) {
					// Original order was within the last 23 hours, so this is a webhook for the first subscription charge, which can be ignored
					if (!empty($webhook['payment_intent']) && !empty($original_order_id)) {
						$order_info = $this->model_checkout_order->getOrder($original_order_id);
						$new_description = $this->replaceShortcodes($settings['transaction_description'], $order_info);
						
						$this->curlRequest('POST', 'payment_intents/' . $webhook['payment_intent'], array('description' => $new_description));
					}
					
					return;
				}
			} else {
				$last_order_query = $this->db->query("SELECT * FROM `" . DB_PREFIX . "order` WHERE email = '" . $this->db->escape($webhook['customer_email']) . "' ORDER BY date_added DESC");
				if ($last_order_query->num_rows && (strtotime($now_query->row['NOW()']) - strtotime($last_order_query->row['date_added'])) < 600) {
					if ($last_order_query->row['user_agent'] != 'Stripe/1.0 (+https://stripe.com/docs/webhooks)') {
						// Customer's last order is within 10 minutes, and is not a Stripe webhook order, so it most likely was an immediate subscription and is already shown on their last order
						return;
					}
				}
			}
			
			// Create order in database
			$this->load->model('extension/' . $this->type . '/' . $this->name);
			
			$order_id = $this->{'model_extension_'.$this->type.'_'.$this->name}->createOrder($data);
			$order_status_id = $settings['success_status_id'];
			
			$strong = '<strong style="display: inline-block; width: 140px; padding: 3px">';
			$comment = $strong . 'Charged for Plan:</strong>' . implode(', ', $plan_ids) . '<br>';
			$comment .= $strong . 'Stripe Event ID:</strong>' . $event['id'] . '<br>';
			
			if (!empty($webhook['payment_intent'])) {
				$comment .= $strong . 'Stripe Payment ID:</strong><a target="_blank" href="https://dashboard.stripe.com/' . ($settings['transaction_mode'] == 'test' ? 'test/' : '') . 'payments/' . $webhook['payment_intent'] . '">' . $webhook['payment_intent'] . '</a><br>';
				
				$order_info = $this->model_checkout_order->getOrder($order_id);
				$new_description = $this->replaceShortcodes($settings['transaction_description'], $order_info);
				
				$this->curlRequest('POST', 'payment_intents/' . $webhook['payment_intent'], array('description' => $new_description));
			}
			
			if (!empty($original_order_id)) {
				$comment .= $strong . 'Original Order ID:</strong>' . $original_order_id . '<br>';
			}
			
			$this->model_checkout_order->addOrderHistory($order_id, $order_status_id, $comment, false);
		}
		
	}
	
	//==============================================================================
	// getSettings()
	//==============================================================================
	private function getSettings() {
		$code = (version_compare(VERSION, '3.0', '<') ? '' : $this->type . '_') . $this->name;
		
		$settings = array();
		$settings_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "setting WHERE `code` = '" . $this->db->escape($code) . "' ORDER BY `key` ASC");
		
		foreach ($settings_query->rows as $setting) {
			$value = $setting['value'];
			if ($setting['serialized']) {
				$value = (version_compare(VERSION, '2.1', '<')) ? unserialize($setting['value']) : json_decode($setting['value'], true);
			}
			$split_key = preg_split('/_(\d+)_?/', str_replace($code . '_', '', $setting['key']), -1, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);
			
				if (count($split_key) == 1)	$settings[$split_key[0]] = $value;
			elseif (count($split_key) == 2)	$settings[$split_key[0]][$split_key[1]] = $value;
			elseif (count($split_key) == 3)	$settings[$split_key[0]][$split_key[1]][$split_key[2]] = $value;
			elseif (count($split_key) == 4)	$settings[$split_key[0]][$split_key[1]][$split_key[2]][$split_key[3]] = $value;
			else 							$settings[$split_key[0]][$split_key[1]][$split_key[2]][$split_key[3]][$split_key[4]] = $value;
		}
		
		return $settings;
	}
	
	//==============================================================================
	// replaceShortcodes()
	//==============================================================================
	private function replaceShortcodes($text, $order_info) {
		$product_names = array();
		
		foreach ($this->cart->getProducts() as $product) {
			$options = array();
			foreach ($product['option'] as $option) {
				$options[] = $option['name'] . ': ' . $option['value'];
			}
			$product_name = $product['name'] . ($options ? ' (' . implode(', ', $options) . ')' : '');
			$product_names[] = html_entity_decode($product_name, ENT_QUOTES, 'UTF-8');
		}
		
		$replace = array(
			'[store]',
			'[order_id]',
			'[amount]',
			'[email]',
			'[comment]',
			'[products]'
		);
		
		$with = array(
			$this->config->get('config_name'),
			$order_info['order_id'],
			$this->currency->format($order_info['total'], $order_info['currency_code']),
			$order_info['email'],
			$order_info['comment'],
			implode(', ', $product_names)
		);
		
		return str_replace($replace, $with, $text);
	}
	
	//==============================================================================
	// metadata()
	//==============================================================================
	private function metadata($order_info) {
		$metadata['Store'] = $this->config->get('config_name');
		$metadata['Order ID'] = $order_info['order_id'];
		$metadata['Customer Info'] = $order_info['firstname'] . ' ' . $order_info['lastname'] . ', ' . $order_info['email'] . ', ' . $order_info['telephone'] . ', customer_id: ' . $order_info['customer_id'];
		$metadata['Products'] = $this->replaceShortcodes('[products]', $order_info);
		$metadata['Order Comment'] = $order_info['comment'];
		$metadata['IP Address'] = $order_info['ip'];
		
		foreach ($metadata as &$md) {
			if (strlen($md) > 497) {
				$md = mb_substr($md, 0, 497, 'UTF-8') . '...';
			}
		}
		
		return $metadata;
	}
	
	//==============================================================================
	// sendEmail()
	//==============================================================================
	private function sendEmail($to, $subject, $message) {
		if (version_compare(VERSION, '2.0.2', '<')) {
			$mail = new Mail($this->config->get('config_mail'));
			$protocol_engine = $mail->protocol;
		} else {
			if (version_compare(VERSION, '3.0', '<')) {
				$mail = new Mail();
				$mail->protocol = $this->config->get('config_mail_protocol');
				$protocol_engine = $this->config->get('config_mail_protocol');
			} else {
				$mail = new Mail($this->config->get('config_mail_engine'));
				$protocol_engine = $this->config->get('config_mail_engine');
			}
			$mail->parameter = $this->config->get('config_mail_parameter');
			$mail->smtp_hostname = $this->config->get('config_mail_smtp_hostname');
			$mail->smtp_username = $this->config->get('config_mail_smtp_username');
			$mail->smtp_password = html_entity_decode($this->config->get('config_mail_smtp_password'), ENT_QUOTES, 'UTF-8');
			$mail->smtp_port = $this->config->get('config_mail_smtp_port');
			$mail->smtp_timeout = $this->config->get('config_mail_smtp_timeout');
		}
		
		if (!is_array($to)) $to = array($to);
		
		foreach ($to as $email) {
			if (empty($email)) continue;
			
			$mail->setSubject($subject);
			$mail->setHtml($message);
			$mail->setText(strip_tags(str_replace('<br>', "\n", $message)));
			$mail->setSender(str_replace(array(',', '&'), array('', 'and'), html_entity_decode($this->config->get('config_name'), ENT_QUOTES, 'UTF-8')));
			$mail->setFrom($this->config->get('config_email'));
			$mail->setTo(trim($email));
			$mail->send();
		}
	}
	
	//==============================================================================
	// curlRequest()
	//==============================================================================
	private function curlRequest($request, $api, $data = array()) {
		$this->load->model('extension/' . $this->type . '/' . $this->name);
		return $this->{'model_extension_'.$this->type.'_'.$this->name}->curlRequest($request, $api, $data);
	}
}
?>